/***************************************************************************
 *
 * Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "GraphicSystems/Gal2dGraphicSystem.h"
#include "WindowSystems/WaylandGal2dWindowSystem.h"
#include "IlmMatrix.h"
#include "string.h"
#include "Bitmap.h"
#include "ViewportTransform.h"
#include "config.h"
#include "Vector2.h"

#include "HAL/gc_hal.h"
#include "HAL/gc_hal_raster.h"
#include "HAL/gc_hal_eglplatform.h"

#include <string>
#include <set>
#include <algorithm>
#include <sys/mman.h>

#include <unistd.h>
#include <sys/syscall.h>
#include <sys/types.h>
#include <errno.h>
#include <libudev.h>

#include <imx_drm.h>
#include "PlatformSurfaces/GcoSurfWaylandPlatformSurface.h"

#include <sys/eventfd.h>

#include "ilm-wl-drm-server-protocol.h"

static const char default_seat[] = "seat0";

static inline void
timespecSub(struct timespec *result,
            const struct timespec *time1, const struct timespec *time2)
{
    result->tv_sec = time1->tv_sec - time2->tv_sec;
    result->tv_nsec = time1->tv_nsec - time2->tv_nsec;
    if (result->tv_nsec < 0) {
        result->tv_sec--;
        result->tv_nsec += NANOSEC_PER_SEC;
    }
}

static inline int64_t
timespecToNsec(const struct timespec *time)
{
    return (int64_t)time->tv_sec * NANOSEC_PER_SEC + time->tv_nsec;
}

class FileThreadObject
{

public:
    FileThreadObject(const char* filename,
                     int size,
                     char* buffer,
                     Gal2dGraphicSystem *graphicSystem,
                     sem_t *workerSem,
                     gcoSURF *tempSurf = NULL)
    : filename(filename)
    , size(size)
    , isLayer(false)
    , format(gcvSURF_UNKNOWN)
    , tempSurf(tempSurf)
    , buffer(buffer)
    , graphicSystem(graphicSystem)
    , workerSem(workerSem)
    {
    }

    FileThreadObject(const char* filename,
                     int width,
                     int height,
                     char* buffer,
                     gceSURF_FORMAT format,
                     Gal2dGraphicSystem *graphicSystem,
                     sem_t *workerSem)
    : filename(filename)
    , size(0)
    , width(width)
    , height(height)
    , isLayer(false)
    , format(format)
    , buffer(buffer)
    , graphicSystem(graphicSystem)
    , workerSem(workerSem)
    {
    }


    FileThreadObject(const char* filename,
                     int width,
                     int height,
                     int blankWidth,
                     int blankHeight,
                     int bottomSize,
                     int bufferDepth,
                     size_t start_offset,
                     size_t display_row_width,
                     size_t layer_row_width,
                     char* buffer,
                     Gal2dGraphicSystem *graphicSystem,
                     sem_t *workerSem)
    : filename(filename)
    , width(width)
    , height(height)
    , blankWidth(blankWidth)
    , blankHeight(blankHeight)
    , bottomSize(bottomSize)
    , bufferDepth(bufferDepth)
    , start_offset(start_offset)
    , display_row_width(display_row_width)
    , layer_row_width(layer_row_width)
    , isLayer(true)
    , format(gcvSURF_UNKNOWN)
    , buffer(buffer)
    , graphicSystem(graphicSystem)
    , workerSem(workerSem)
    {
    }

    ~FileThreadObject()
    {
        delete[] filename;
    }

    void exec()
    {
        LOG_DEBUG("FileThreadObject", "Writing file....");
        if (isLayer)
        {
            graphicSystem->writeLayerFile(filename, width,
                                          height, blankWidth,
                                          blankHeight, bottomSize, bufferDepth,
                                          start_offset, display_row_width,
                                          layer_row_width, buffer);
        }
        else if (size != 0)
        {
            graphicSystem->writeFile(filename, size, buffer, tempSurf);
        }
        else
        {
            graphicSystem->writeBMPFile(filename, width, height, format, buffer);
        }
        sem_post(workerSem);
    }

    static void *run(void *ptr)
    {
        FileThreadObject *thread = static_cast<FileThreadObject*>((FileThreadObject*) ptr);
        LOG_DEBUG("FileThreadObject", "Run called");
        thread->exec();
        pthread_exit(NULL);
    }

private:
    const char* filename;
    int size;
    int width;
    int height;
    int blankWidth;
    int blankHeight;
    int bottomSize;
    int bufferDepth;
    size_t start_offset;
    size_t display_row_width;
    size_t layer_row_width;
    bool isLayer;
    gceSURF_FORMAT format;
    gcoSURF *tempSurf;
    char *buffer;
    Gal2dGraphicSystem *graphicSystem;
    sem_t *workerSem;


};

Gal2dGraphicSystem::Gal2dGraphicSystem(int windowWidth, int windowHeight, Configuration* config)
: m_windowWidth(windowWidth)
, m_windowHeight(windowHeight)
, m_nativeDisplay(0)
, m_nativeWindow(0)
, m_displayWidth(0)
, m_displayHeight(0)
, m_currentLayer(0)
, m_hw2DPE20(gcvFALSE)
, m_gcoOs(0)
, m_gcoHal(0)
, m_gco2dEngine(0)
, m_localDisplayInfo(0)
, m_drm_master_sem(NULL)
, m_fdDev(-1)
, m_crtcs(NULL)
, m_crtcsNum(0)
, m_currentOutput(NULL)
, m_availableCrtcs(0)
, m_usedCrtcs(0)
, dpms_fdDev(-1)
, m_currentScreenshotThread(NULL)
, m_fileThread(0)
, mConfiguration(config)
{
	LOG_DEBUG("Gal2dGraphicSystem", "Creating Gal2dGraphicSystem");
    for (int i=0; i < OPT_COUNT; i++)
    {
        m_optimizations[i] = OPT_MODE_HEURISTIC;
    }
    m_dualplain_crtcs[0]=-1;
    m_dualplain_crtcs[1]=-1;

    pthread_mutexattr_t attr;
    pthread_mutexattr_init(&attr);
    /*
      * multiple times the mutex will be acquired in the code flow ,
      * so using recursive type mutex
      */
    pthread_mutexattr_settype (&attr, PTHREAD_MUTEX_RECURSIVE_NP);
    pthread_mutex_init(&m_graSystemMutex, &attr);
}

Gal2dGraphicSystem::~Gal2dGraphicSystem()
{
    LOG_DEBUG("Gal2dGraphicSystem", "Destructor");
    pthread_mutex_destroy(&m_graSystemMutex);
    releaseResources();
}

VivTile Gal2dGraphicSystem::validTileSizes[MAX_TILING_LEVEL] = {
							 { 0, 0, 64, 64, 32, 32, 1},
							 { 64, 64, 32, 32, 16, 16, 2},
							 { 32, 32, 16, 16, 4, 4, 2},
							 { 16, 16, 4,  4,  0, 0, 2}};


unsigned int* Gal2dGraphicSystem::ReadPixelValue(unsigned int *src, unsigned int x, unsigned int y, unsigned int tilingLevel)
{
	VivTile *viv_tile;
	unsigned int tileGroupsPerRow;
	unsigned int tileGrpNumOfCord;
	unsigned int tileNumOfCord;
	unsigned int tileSize;
	unsigned int rowOfTileGrp;
	unsigned int colOfTileGrp;
	unsigned int rowOfTile;
	unsigned int colOfTile;
	unsigned int tileGroupsize;
	unsigned int tileGrp_x, tile_x;
	unsigned int tileGrp_y, tile_y;

	viv_tile = &validTileSizes[tilingLevel];

	tileSize = viv_tile->tileW * viv_tile->tileH;

	/*tile group calculation*/
	tileGroupsize = (viv_tile->tileGroupDim * viv_tile->tileGroupDim);
	tileGroupsPerRow = (viv_tile->prevTileW / (viv_tile->tileGroupDim * viv_tile->tileW));
	tileGrpNumOfCord = ((tileGroupsPerRow * (y / (viv_tile->tileGroupDim * viv_tile->tileH))) + (x / (viv_tile->tileGroupDim * viv_tile->tileW)));

	src = &src[tileGrpNumOfCord * tileGroupsize * tileSize];/*move source address to tileGroup*/

	rowOfTileGrp = (tileGrpNumOfCord / tileGroupsPerRow);
	colOfTileGrp = tileGrpNumOfCord - (rowOfTileGrp * tileGroupsPerRow);
	tileGrp_x = colOfTileGrp * viv_tile->tileGroupDim * viv_tile->tileW;
	tileGrp_y = rowOfTileGrp * viv_tile->tileGroupDim * viv_tile->tileH;
	x = x - tileGrp_x;
	y = y - tileGrp_y;

	/*tile calculation*/
	tileNumOfCord = (viv_tile->tileGroupDim * (y / viv_tile->tileH) + (x / viv_tile->tileW));

	src = &src[tileNumOfCord * tileSize];/*move source address to tile inside the tileGroup*/

	rowOfTile = (tileNumOfCord / viv_tile->tileGroupDim);
	colOfTile = tileNumOfCord - (rowOfTile * viv_tile->tileGroupDim);
	tile_x = colOfTile * viv_tile->tileW;
	tile_y = rowOfTile * viv_tile->tileH;
	x = x - tile_x;
	y = y - tile_y;

	if ((viv_tile->nextTileH != 0) && (viv_tile->nextTileW != 0))
	{
		tilingLevel++;
		src = ReadPixelValue(src, x, y, tilingLevel);

	}
	else
	{
		src = &src[(y * viv_tile->tileW) + (x)];/*move source address to pixel inside the tile*/
	}

	return src;

}

bool Gal2dGraphicSystem::isSurfaceTransparentOn(Surface* surf, int x , int y)
{
    bool retv = false;
    gceTILING tiling;

    GcoSurfWaylandPlatformSurface* nativeSurface = (GcoSurfWaylandPlatformSurface*)surf->platform;
    if (!nativeSurface || !nativeSurface->m_surface)
    {
        LOG_WARNING("Gal2dGraphicSystem", "bad nativeSurface for surface ilm ID: "<<surf->getID());
        retv = false;
    }
    else
    {
        unsigned char *pixel;
        gceSTATUS status = gcvSTATUS_OK;
        gctPOINTER logAddr[3] = {NULL, NULL, NULL};
        gcsSURF_FORMAT_INFO_PTR formatInfo = 0;
        gctUINT width = 0;
        gctUINT height = 0;
        gctINT stride = 0;

        status = gcoSURF_Lock(nativeSurface->m_surface,NULL, (gctPOINTER*)&logAddr);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to lock surface, "
                        "gcoSURF=" << nativeSurface->m_surface);
            retv = false;
            return retv;
        }

        status = gcoSURF_GetFormatInfo(nativeSurface->m_surface, &formatInfo);
        gcoSURF_GetTiling(nativeSurface->m_surface, &tiling);

        LOG_DEBUG("Gal2dGraphicSystem",
                  "Tiling Format, "
                  "surface=" << surf << ", "
                  "surface ID=" << surf->getID() << ", "
                  "Tiling=" << tiling);


        if (gcvSTATUS_OK == status && formatInfo != 0 && formatInfo->fmtClass == gcvFORMAT_CLASS_RGBA)
        {
            if ((formatInfo->u.rgba.alpha.width > 0) &&
                  ((formatInfo->u.rgba.alpha.width & gcvCOMPONENT_DONTCARE) == 0))
            {
                status = gcoSURF_GetAlignedSize(nativeSurface->m_surface, &width, &height, &stride);

                if (gcvSTATUS_OK == status &&
                        y < (int)height && x < (int)width)
                {
                    /*alpha channel exist, check the value*/
                    unsigned char *buffer = (unsigned char *)logAddr[0];
                    int offset = 0;
                    if ((buffer != NULL) && (gcvLINEAR == tiling))
                    {
                        offset = (y * stride) + (x * (stride/width));
                        pixel = (unsigned char *)&buffer[offset];
                        if (pixel[3] == 0)
                            retv = true;
                        else
                            retv = false;

                        LOG_DEBUG("Gal2dGraphicSystem", " surface ilm ID: "<<surf->getID()
                                <<" on: ("<< x <<", "<< y <<") "
                                <<"has value:" <<(int)pixel[0]<<" "<<(int)pixel[1]<<" " <<(int)pixel[2]<<" "<<(int)pixel[3]);
                    }
                    else if((buffer != NULL) && (gcvSUPERTILED == tiling))
                    {
                        /*64x64 SUPERTILING: In case of supertiled buffer, a tile of 64x64 pixel is
                         * recursively tiled to 32x32, 16x16 and 4x4 pixel tiles.
                         * Every 32x32, 16x16 and 8x8 tiles are arranged in a Z-order.
                         * The outermost 64x64 tiles are placed linearly and not in Z-order
                         *
                         * Z-ordering: https://en.wikipedia.org/wiki/Z-order_curve
                         */

                        validTileSizes[0].prevTileW = width;
                        validTileSizes[0].prevTileH = height;

                        pixel = (unsigned char*)ReadPixelValue((unsigned int*)logAddr[0], x, y,(unsigned int)0);
                        if (pixel[3] == 0)
                        {
                            retv = true;
                        }
                        else
                        {
                            retv = false;
                        }

                    LOG_DEBUG("Gal2dGraphicSystem", " surface ilm ID: "<<surf->getID()
                                <<" on: ("<< x <<", "<< y <<") "
					            <<"has value:" <<(int)pixel[0]<<" "<<(int)pixel[1]<<" " <<(int)pixel[2]<<" "<<(int)pixel[3]);

                    }
                    else
                    {
                        LOG_WARNING("Gal2dGraphicSystem",
                                    "buffer address is NULL");
                        retv =false;
                    }
                }
                else
                {
                    LOG_WARNING("Gal2dGraphicSystem",
                                "Failed to get aligned size, "
                                "status=" << status);
                    retv =false;
                }

            }
            else
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "gcoSURF=" << nativeSurface->m_surface
                            << " ilm id = " << surf->getID()
                            << " doesn't have a alpha channel");
            }
        }
        else
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Color format is other than RGBA. "
                        "Tiling not supported");
        }

        status = gcoSURF_Unlock(nativeSurface->m_surface, (gctPOINTER)logAddr);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to unlock surface, "
                        "gcoSURF=" << nativeSurface->m_surface);
        }
    }

    return retv;
}
void Gal2dGraphicSystem::releaseResources()
{
    struct DrmOutput* output,*next = NULL;
    struct drm_mode_destroy_dumb dest_req;
    unsigned int ii = 0;
    int ret = 0;
    int errorNo = 0;

    wl_list_for_each_safe(output,next, &m_outputList, link)
    {
        if (NULL != output->dstSurfaces)
        {
            delete[] output->dstSurfaces;
        }

        errno = 0;
        if (output->orgCrtc->mode_valid)
        {
            //restore saved screen configuration
            ret = drmModeSetCrtc(m_fdDev,
                           output->orgCrtc->crtc_id,
                           output->orgCrtc->buffer_id,
                           output->orgCrtc->x,
                           output->orgCrtc->y,
                           &output->connectorID,
                           1,
                           &output->orgCrtc->mode);
            errorNo = errno;
            if (ret)
            {
                char errbuf[BUFSIZ];
                strerror_r(errorNo, errbuf, BUFSIZ);

                LOG_WARNING("Gal2dGraphicSystem",
                            "Failed to restore crtc, screen="
                            << output->screenID << ", Error=" << errbuf << ", "
                            "CRTC ID=" << output->orgCrtc->crtc_id << ", "
                            "buffer ID=" << output->orgCrtc->buffer_id << ", "
                            "connector ID=" << output->connectorID);
            }
            else
            {
                LOG_DEBUG("Gal2dGraphicSystem",
                          "Restored crtc, screen=" << output->screenID << ", "
                          "CRTC ID=" << output->orgCrtc->crtc_id << ", "
                          "buffer ID=" << output->orgCrtc->buffer_id << ", "
                          "connector ID=" << output->connectorID);
            }
        }
        if (output->propDPMS)
        	drmModeFreeProperty(output->propDPMS);
        drmModeFreeCrtc(output->orgCrtc);

        for(ii = 0; ii < output->dstNumSurfaces; ii++)
        {
            munmap(output->buffer[ii].pixels, output->buffer[ii].size);
            errno = 0;
            ret = drmModeRmFB(m_fdDev, output->buffer[ii].fb);
            errorNo = errno;
            if (ret)
            {
                char errbuf[BUFSIZ];
                strerror_r(errorNo, errbuf, BUFSIZ);

                LOG_WARNING("Gal2dGraphicSystem",
                            "Failed to remove FB of screen="
                            << output->screenID << ", "
                            "error=" << errbuf);
            }
            else
            {
                LOG_DEBUG("Gal2dGraphicSystem",
                          "Removed FB of screen=" << output->screenID);
            }
            /* free dumb buffer */
            memset(&dest_req, 0, sizeof(dest_req));
            dest_req.handle = output->buffer[ii].handle;
            errno = 0;
            ret = drmIoctl(m_fdDev, DRM_IOCTL_MODE_DESTROY_DUMB, &dest_req);
            errorNo = errno;
            if (ret)
            {
                char errbuf[BUFSIZ];
                strerror_r(errorNo, errbuf, BUFSIZ);

                LOG_WARNING("Gal2dGraphicSystem",
                            "Failed to destroy drm buffer(ID="
                            << output->buffer[ii].handle << ") "
                            "of screen="<< output->screenID << ", "
                            "error=" << errbuf);
            }
            else
            {
                LOG_DEBUG("Gal2dGraphicSystem",
                          "Successfully destroyed drm buffer(ID="
                          << output->buffer[ii].handle
                          << ") of screen="<< output->screenID);
            }
        }
        delete[] output->buffer;
        wl_list_remove(&output->link);
        free(output);
    }

    wl_list_for_each_safe(output,next, &m_screenshotList, link)
    {
        if (NULL != output->dstSurfaces)
        {
            gcoSURF_Destroy(output->dstSurfaces[output->currentIndex]);
            delete[] output->dstSurfaces;
        }
        delete[] output->buffer;
        wl_list_remove(&output->link);
        free(output);
    }

    if (m_currentScreenshotThread != NULL)
    {
        pthread_join(m_fileThread, NULL);
        delete m_currentScreenshotThread;
    }

    if (gcvNULL != m_gcoHal)
    {
        gco2D_Flush(m_gco2dEngine);
        gcoHAL_Commit(m_gcoHal, gcvTRUE);
    }

    if (gcvNULL != m_gco2dEngine)
    {
        gco2D_FreeFilterBuffer(m_gco2dEngine);
        m_gco2dEngine = gcvNULL;
    }

    if (gcvNULL != m_gcoHal)
    {
        gcoHAL_Destroy(m_gcoHal);
        m_gcoHal = gcvNULL;
    }

    if (gcvNULL != m_gcoOs)
    {
        gcoOS_Destroy(m_gcoOs);
        m_gcoOs = gcvNULL;
    }

    if (NULL != m_nativeDisplay)
    {
        gcoOS_DestroyDisplay(m_nativeDisplay);
        m_nativeDisplay = NULL;
    }

    if (NULL != m_crtcs)
    {
        free(m_crtcs);
        m_crtcs = NULL;
    }

    if (m_fdDev >= 0 )
    {
        close(m_fdDev);
    }

    if (dpms_fdDev >= 0 )
    {
        close(dpms_fdDev);
    }

    if (m_drm_master_sem != NULL)
    {
        sem_post(m_drm_master_sem);
        sem_close(m_drm_master_sem);
        sem_unlink("/drm_master");
    }
}

int Gal2dGraphicSystem::getDepthFromGcoSurfFormat(gceSURF_FORMAT format)
{
   int depth = 0;
   switch(format)
   {
       case gcvSURF_R5G6B5:
       case gcvSURF_B5G6R5:
           depth = 16;
           break;
       case gcvSURF_A8R8G8B8:
       case gcvSURF_X8R8G8B8:
       case gcvSURF_R8G8B8A8:
       case gcvSURF_R8G8B8X8:
       case gcvSURF_X8B8G8R8:
       case gcvSURF_A8B8G8R8:
       case gcvSURF_B8G8R8X8:
       case gcvSURF_B8G8R8A8:
           depth = 32;
           break;
       default:
           depth = 0;
   }

   return depth;
}
void Gal2dGraphicSystem::setTargetSurface(gcoSURF surface)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called, gcoSURF=" << surface);
    gceSTATUS status = gcvSTATUS_OK;
    gctUINT width = 0;
    gctUINT height = 0;
    gctINT stride = 0;
    gctUINT32 phyAddr = 0;
    gctPOINTER logAddr = NULL;
/*
    if (m_currentOutput->currentTargetSurface == surface)
    {
        return;
    }
*/
    status = gcoSURF_GetAlignedSize(surface, &width, &height, &stride);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get native surface aligned size, "
                    "gcoSURF=" << surface);
        return;
    }

    status = gcoSURF_Lock(surface, &phyAddr, (gctPOINTER*)&logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to lock surface, "
                    "gcoSURF=" << surface);
        return;
    }
    status = gco2D_SetTargetEx(m_gco2dEngine, phyAddr, stride, gcvSURF_0_DEGREE, width, height);

    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to set target surface, "
                    "gcoSURF=" << surface);
    }

    status = gcoSURF_Unlock(surface, (gctPOINTER)logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to unlock surface, "
                    "gcoSURF=" << surface);
    }

    LOG_DEBUG("Gal2dGraphicSystem",
              "Set target surface, "
              "gcoSURF=" << surface);

    m_currentOutput->currentTargetSurface = surface;
}

void Gal2dGraphicSystem::activateGraphicContext()
{
    LOG_DEBUG("Gal2dGraphicSystem","Called");
}

DrmOutput* Gal2dGraphicSystem::findScreenshotOutput(uint screenID)
{
    struct DrmOutput* output = NULL;
    wl_list_for_each(output, &m_screenshotList, link)
    {
        if (output->screenID != screenID)
        {
            continue;
        }
        return output;
    }
    return NULL;
}

bool Gal2dGraphicSystem::switchScreen(uint screenID, bool takeScreenshot)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called, screen ID=" << screenID);
    // Actually, when renderSWLayers is called, rendering target buffer is switched
    // because of avoiding overhead of switching display.
    gceSTATUS status = gcvSTATUS_OK;
    struct DrmOutput* output = NULL;
    if (!takeScreenshot)
    {
        wl_list_for_each(output, &m_outputList, link)
        {
            if (output->screenID != screenID)
            {
                continue;
            }

            LOG_DEBUG("Gal2dGraphicSystem",
                      "Switch screen, ID=" << m_currentOutput->screenID);

            m_currentOutput = output;
            break;
        }
    }
    else
    {
        if (sem_trywait(&m_workerSem) < 0) {
            LOG_DEBUG("Gal2dGraphicSystem",
                      "We are currently writing a screenshot, aborting");
            return false;
        }

        output = findScreenshotOutput(screenID);

        if (output != NULL)
            m_currentOutput = output;
        else
            return false;
    }
    setTargetSurface(m_currentOutput->dstSurfaces[m_currentOutput->currentIndex]);

    if  (m_currentOutput->depth == 16)
    {
        status = gco2D_EnableDither( m_gco2dEngine, m_currentOutput->dither);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Dithering for screen:" << screenID << " could not be set to " << m_currentOutput->dither);
        }
    }
    else
    {
        status = gco2D_EnableDither( m_gco2dEngine,gcvFALSE);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Dithering for 2D core on screen "<<screenID<<" could not be disabled");
        }
    }

    resize(m_currentOutput->currentMode->width, m_currentOutput->currentMode->height);
    return true;
}

uint32_t
Gal2dGraphicSystem::getScreenIdFromConnectorType(uint32_t connector_type,uint32_t connector_type_id)
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, connector type=" << connector_type);
    if (connector_type == DRM_MODE_CONNECTOR_HDMIA)
        return mConfiguration->getHdmIid();

    if (connector_type == DRM_MODE_CONNECTOR_LVDS)
    {
        if (connector_type_id == 1)
            return mConfiguration->getLvds0id();
        if (connector_type_id == 2)
            return mConfiguration->getLvds1id();
    }
    if (connector_type == DRM_MODE_CONNECTOR_VGA)
        return mConfiguration->getParallelid();

    if (connector_type == DRM_MODE_CONNECTOR_DSI)
        return mConfiguration->getDsi0id();

    LOG_WARNING("Gal2dGraphicSystem",
                "Unknown DRM connector type=" << connector_type
                << ", connector ID=" << connector_type_id);

    return ILM_DISPLAY_UNKNOWN;
}

static bool
setDpmsState(void *data)
{
    LOG_DEBUG("Gal2dGraphicSystem","called");
    struct DrmOutput *output = static_cast<struct DrmOutput*>(data);
    bool success = false;
    int ret = -1;

	LOG_DEBUG("Gal2dGraphicSystem", "calling drmModeConnectorSetProperty");
	ret = drmModeConnectorSetProperty(output->fdDev, output->connectorID,
					output->propDPMS->prop_id, output->dpms_newState);

	if (!ret)
	{
        output->dpms_current_state= output->dpms_newState;
        output->dpms_readytoChange = false;

        if(output->dpms_current_state == DRM_MODE_DPMS_OFF)
        {
            /* ilm to render in previous index during DPMS OFF state,
                making current index to previous index */
            if(output->currentIndex == 0)
            {
                output->currentIndex = output->dstNumSurfaces - 1;
            }
            else
            {
                output->currentIndex = output->currentIndex - 1;
            }
        }
        else if(output->dpms_current_state == DRM_MODE_DPMS_ON)
        {
            output->currentIndex = (output->currentIndex + 1) %
                                                output->dstNumSurfaces;
        }
        else
        {
            /* have to implement for STANDBY and SUSPEND */
        }
        success = true;
	}
	else
	{
		LOG_ERROR("Gal2dGraphicSystem",
				"failed to set dpms property for screen = " << output->screenID
				<< " connector_id = " << output->connectorID <<
				 " error=" << ret);
	}

    LOG_DEBUG("Gal2dGraphicSystem", "Exited");
    return success;
}


void Gal2dGraphicSystem::pageFlipHandler(int /*fd*/, unsigned int /*frame*/, unsigned int sec,
                                unsigned int usec, void *data)
{
    struct DrmOutput *output = static_cast<struct DrmOutput*>(data);
    struct timespec tFlip;

    /*we don't sent pageFlipPending in startRepaintLoop*/
    if (output->pageFlipPending) {
        output->currentIndex = (output->currentIndex + 1) %
                                                output->dstNumSurfaces;
    }

    output->pageFlipPending = false;

    if (output->windowSystem)
    {
        WaylandBaseWindowSystem* windowSystem;
        windowSystem = static_cast<WaylandBaseWindowSystem*>(output->windowSystem);

        tFlip.tv_sec = sec;
        tFlip.tv_nsec = usec * 1000;

        LOG_DEBUG("Gal2dGraphicSystem","PFlip Time: sec="<< tFlip.tv_sec << ", " <<
                  "nsec=" << tFlip.tv_nsec);

        windowSystem->updateRepaintTimer(output->screenID, &tFlip, output->repaintWindowPeriodMS,
                output->currentMode->refresh_ns, REPAINT_TIMER_UPDATE_PAGE_FLIP);
    }

    output->graphicSystem->setDpmsUponStateChange(output);
}

int Gal2dGraphicSystem::onDrmInput(int fd, uint32_t /*mask*/, void * /*data*/)
{
        LOG_DEBUG("Gal2dGraphicSystem","Called, fd=" << fd);
        drmEventContext evctx;
        memset(&evctx, 0, sizeof evctx);

        evctx.version = DRM_EVENT_CONTEXT_VERSION;
        evctx.page_flip_handler = pageFlipHandler;
        evctx.vblank_handler = NULL;

        drmHandleEvent(fd, &evctx);

        return 1;
}

int Gal2dGraphicSystem::setDpmsUponStateChange(struct DrmOutput *output)
{
	int ret = 0;
	pthread_mutex_lock(&m_graSystemMutex);
	if (NULL == output)
	{
        wl_list_for_each(output, &m_outputList, link)
        {
            if((output->dpms_readytoChange == true) && !(output->pageFlipPending))
            {
                ret = setDpmsState(output);
            }
        }
	}
	else
	{
	    if(output->dpms_readytoChange == true)
	    {
	        ret = setDpmsState(output);
	    }
	}
	pthread_mutex_unlock(&m_graSystemMutex);
	return ret;
}

int Gal2dGraphicSystem::onDpmsInput(int fd, uint32_t /*mask*/, void *data)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called, fd=" << fd);

    int ret = -1;
    uint64_t fd_value;
    ret = read(fd, &fd_value, sizeof(uint64_t));
    if(ret == -1)
    {
        LOG_ERROR("Gal2dGraphicSystem", "FAILED to read from dpms_fdDev ");
    }
    else
    {
        Gal2dGraphicSystem* graSystem = static_cast<Gal2dGraphicSystem*>(data);
        graSystem->setDpmsUponStateChange(NULL);
    }

    LOG_DEBUG("Gal2dGraphicSystem", "Exited");
    return ret;
}

bool Gal2dGraphicSystem::initializeSystem()
{
    bool ret = true;
    LOG_DEBUG("Gal2dGraphicSystem", "Called");

    WaylandBaseWindowSystem* windowSystem = dynamic_cast<WaylandBaseWindowSystem*>(m_baseWindowSystem);
    struct wl_display* wlDisplay = windowSystem->getNativeDisplayHandle();

    if (!createOutputs())
        ret = false;

    struct wl_event_loop *loop = wl_display_get_event_loop(wlDisplay);
    wl_event_loop_add_fd(loop, m_fdDev, WL_EVENT_READABLE, Gal2dGraphicSystem::onDrmInput, NULL);

    dpms_fdDev = eventfd(0, EFD_CLOEXEC);
    wl_event_loop_add_fd(loop, dpms_fdDev, WL_EVENT_READABLE, Gal2dGraphicSystem::onDpmsInput, this);

    LOG_DEBUG("Gal2dGraphicSystem", "Exited");
    return ret;
}

void Gal2dGraphicSystem::sort_dualplain_crtcs( int num_crtcs)
{
    int crtc = -1;
    int current_crtc = 0;

    if (num_crtcs == 2)
    {
        /*check if connector is free to use*/
        crtc = m_availableCrtcs & (1 << m_dualplain_crtcs[num_crtcs-1]);
        if (crtc == 0)
        {
            crtc = m_dualplain_crtcs[0];
            m_dualplain_crtcs[0] = m_dualplain_crtcs[num_crtcs-1];
            m_dualplain_crtcs[num_crtcs-1] = crtc;
        }
    }
}

bool Gal2dGraphicSystem::createOutputs()
{
    LOG_DEBUG("Gal2dGraphicSystem", "Called");

    drmModeConnector* connector_list = NULL;
    drmModeConnector* connector = NULL;
    drmModeRes* resources = NULL;
    drmModePlaneRes *plane_resources = NULL;
    drmModePlane *plane = NULL;
    struct ConnectorCrtcAssignment* preferredCrtsForConn;

    bool ret = true;
    int x = 0, y = 0;
    int ii,j;
    int connNum;
    unsigned int cnt_planes = 0;
    m_availableCrtcs = 0;
    m_usedCrtcs = 0;
    bool dualPlaneCrtcCheck = false;
    int dualPlainCrtcNum = 0;
    uint32_t connectorDispID = ILM_DISPLAY_UNKNOWN;
    int preferredCrtc = -1;

    int primCSCdispID = mConfiguration->getPrimaryCSCdisplayID();
    int secondCSCdispID = mConfiguration->getSecondaryCSCdisplayID();

    if ( (primCSCdispID >= 0) || (secondCSCdispID >= 0))
    {
        dualPlaneCrtcCheck = true;
    }

    resources = drmModeGetResources(m_fdDev);

    if (!resources)
    {
        LOG_ERROR("Gal2dGraphicSystem", "drmModeGetResources failed. errno: "<< errno);
        ret = false;
        goto err;
    }

    m_crtcs = (drmModeCrtcPtr*) calloc(resources->count_crtcs, sizeof(drmModeCrtcPtr));
    preferredCrtsForConn = (struct ConnectorCrtcAssignment*)malloc (sizeof(struct ConnectorCrtcAssignment)*
                                                                     resources->count_connectors);

    if (!m_crtcs || !preferredCrtsForConn)
    {
        LOG_ERROR("Gal2dGraphicSystem","Failed to allocate memory for "
                << resources->count_crtcs <<" crtcs or preferredCrtsForConn");
        ret = false;
        goto free_drm_resources;
    }
    m_crtcsNum = resources->count_crtcs;
    connNum = resources->count_connectors;

    memcpy(m_crtcs, resources->crtcs, sizeof(uint32_t) * m_crtcsNum);

    for (ii = 0; ii < m_crtcsNum; ii++)
    {
    	m_crtcs[ii] = drmModeGetCrtc(m_fdDev, resources->crtcs[ii]);
    	if (m_crtcs[ii] == NULL)
    		continue;
    	drmModeCrtc* crtc = (drmModeCrtc*)m_crtcs[ii];

        /*crtc is only available for us if its mode is invalid
         *otherwise it is already used*/
        if ( !crtc->mode_valid )
            m_availableCrtcs |= (1 << crtc->crtc_id);
    }
/*check for the dualplane crtcs*/
    if (dualPlaneCrtcCheck)
    {
        plane_resources = drmModeGetPlaneResources(m_fdDev);
        if (!plane_resources)
        {
            LOG_ERROR("Gal2dGraphicSystem"," drmModeGetPlaneResources failed");
            ret = false;
            goto free_plane_resource;
        }

        for (cnt_planes = 0; cnt_planes < plane_resources->count_planes;
                cnt_planes++)
        {
            plane = drmModeGetPlane(m_fdDev,
                    plane_resources->planes[cnt_planes]);
            if (plane != NULL)
            {
                for (j = 0; j <  m_crtcsNum; j++)
                {
                    if (plane->possible_crtcs & (1 << j))
                    {
                        if ( (m_crtcs[j] != NULL) &&
                                (dualPlainCrtcNum < MAX_DUALPLAIN_CRTCS))
                        {
                            m_dualplain_crtcs[dualPlainCrtcNum] = m_crtcs[j]->crtc_id;
                            dualPlainCrtcNum++;
                            LOG_INFO("Gal2dGraphicSystem","DUALPLAIN CRTC: "<< m_crtcs[j]->crtc_id);
                        }
                    }
                }
                drmModeFreePlane(plane);
            }
            else
            {
                LOG_ERROR("Gal2dGraphicSystem","Failed to retrieve plane");
                ret = false;
                goto free_plane_resource;
            }
        }
    }

    /*we should take unused dual plain crtc first*/
    sort_dualplain_crtcs( dualPlainCrtcNum);

    for (ii = 0; ii < connNum ; ii++)
    {
        connector = drmModeGetConnector(m_fdDev, resources->connectors[ii]);
        if (connector == NULL)
            continue;

        preferredCrtsForConn[ii].drmConnector = NULL;
        preferredCrtsForConn[ii].ilmDispID = NULL;
        preferredCrtsForConn[ii].initialized = false;

        if (connector->connection == DRM_MODE_CONNECTED && connector->count_modes > 0)
        {
            connectorDispID = getScreenIdFromConnectorType( connector->connector_type,
                                                            connector->connector_type_id);
            preferredCrtsForConn[ii].drmConnector = connector;
            preferredCrtsForConn[ii].ilmDispID = connectorDispID;
            preferredCrtc = -1;

            /*assign dual plain crtc if requested and available*/
            if ( (primCSCdispID == connectorDispID) ||
                    (secondCSCdispID == connectorDispID) )
            {
                if (dualPlainCrtcNum > 0)
                {
                    preferredCrtc = m_dualplain_crtcs[dualPlainCrtcNum-1];
                    dualPlainCrtcNum--;
                }
                preferredCrtsForConn[ii].preferredCrtcID = preferredCrtc;

                if (createOutputForConnector(connector,connectorDispID, preferredCrtc) < 0)
                {
                    LOG_ERROR("Gal2dGraphicSystem","could not create output for"
                            "display id: "<<connectorDispID);
                    drmModeFreeConnector(connector);
                    preferredCrtsForConn[ii].drmConnector = NULL;
                    preferredCrtsForConn[ii].ilmDispID = ILM_DISPLAY_UNKNOWN;
                    preferredCrtsForConn[ii].initialized = false;
                }
                else
                {
                    preferredCrtsForConn[ii].initialized = true;
                }
            }
        }

    }

    /*perform initialization for connectors which are not require
     * dual plain crtc*/
    for (ii = 0; ii < connNum ; ii++)
    {
        if (preferredCrtsForConn[ii].drmConnector &&
                (preferredCrtsForConn[ii].initialized == false))
        {
            if (createOutputForConnector(preferredCrtsForConn[ii].drmConnector,
                                         preferredCrtsForConn[ii].ilmDispID,
                                         -1))
            {
                LOG_ERROR("Gal2dGraphicSystem","could not create output for"
                                            "display id: "<<preferredCrtsForConn[ii].ilmDispID);
            }
        }
    }

    for (ii = 0; ii < connNum ; ii++)
    {
        if (!preferredCrtsForConn[ii].drmConnector)
            drmModeFreeConnector(preferredCrtsForConn[ii].drmConnector);
    }

    if (wl_list_empty(&m_outputList))
    {
        LOG_ERROR("Gal2dGraphicSystem", "DrmOutput list is empty.");
        ret = false;
    }

free_plane_resource:
    drmModeFreePlaneResources(plane_resources);
    free(preferredCrtsForConn);

free_drm_resources:
    drmModeFreeResources(resources);

err:
    LOG_DEBUG("Gal2dGraphicSystem", "Exited");
    return ret;
}

int Gal2dGraphicSystem::createOutputForScreenshot(DrmOutput *screen)
{
    gceSTATUS status = gcvSTATUS_OK;
    DrmOutput* output = (DrmOutput*)malloc(sizeof *output);

    memset(output, 0x00, sizeof *output);

    output->screenID = screen->screenID;
    output->windowSystem = m_baseWindowSystem;
    output->currentMode = screen->currentMode;
    output->dstNumSurfaces = 1;
    output->surfaceFormat = screen->surfaceFormat;
    output->dstSurfaces = new gcoSURF[output->dstNumSurfaces];
    output->buffer = new struct drm_buffer[output->dstNumSurfaces];
    output->isScreenshot = true;
    output->currentIndex = 0;
    output->screen_alpha = 1.0f;

    int bufferSize = screen->buffer[output->currentIndex].size;
    int stride = screen->buffer[output->currentIndex].stride;
    struct drm_buffer *buffer = &output->buffer[output->currentIndex];

    buffer->width = output->currentMode->width;
    buffer->height = output->currentMode->height;

    buffer->stride = stride;
    buffer->size = bufferSize;
    buffer->pixels = 0;

    status = gcoSURF_Construct(m_gcoHal,
                               buffer->width,
                               buffer->height, 1,
                               gcvSURF_BITMAP,
                               output->surfaceFormat,
                               gcvPOOL_DEFAULT,
                               &output->dstSurfaces[output->currentIndex]);

    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to create gcoSURF, "
                    "screenshot screen ID=" << output->screenID << ", "
                    "gcoSURF="
                    << output->dstSurfaces[output->currentIndex]);
        return -1;
    }

    LOG_DEBUG("Gal2dGraphicSystem",
              "Buffer Created - "
              "target dstNumSurfaces=" << output->dstNumSurfaces << ", "
              "target surface format=" << output->surfaceFormat << ", "
              "target CRTC ID=" << output->crtcID << ", "
              "connector ID=" << output->connectorID << ", "
              "screenshot screen ID=" << output->screenID << ", "
              "gcoSURF="
              << output->dstSurfaces[output->currentIndex] << ", "
              "width="
              << buffer->width << ", "
              "height=" << buffer->height);

    wl_list_insert(m_screenshotList.prev, &output->link);

    return 0;
}


drmModePropertyPtr
Gal2dGraphicSystem::getDRMProperty(int drmFd,
						drmModeConnector* connector, char *pName)
{
	drmModePropertyPtr props;
	drmModePropertyPtr retProps = NULL;
	int propNum;

	for (propNum = 0;
		(propNum < connector->count_props) && (NULL == retProps);
		 propNum++)
	{
		props = drmModeGetProperty(drmFd, connector->props[propNum]);
		if (NULL != props)
		{
			if (!strcmp(props->name, pName))
			{
				LOG_DEBUG("Gal2dGraphicSystem", "getDRMPropery successful\n ");
				retProps = props;
			}
			else
			{
				drmModeFreeProperty(props);
			}
		}
	}
	return retProps;
}

int Gal2dGraphicSystem::createOutputForConnector(drmModeConnector* connector,
                                                   uint32_t connectorDispID,
                                                   int preferred_crtc)
{

    drmModeEncoder* encoder = NULL;
    struct DrmMode* drmMode = NULL;
    struct DrmOutput *mode = NULL;
    struct wl_list *ptr = NULL;

    gceSTATUS status = gcvSTATUS_OK;
    int ii;
    int k;
    int found_crtc = 0;
    int ret;
    unsigned int screen_id = 0;

    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, connector ID="
              << connector->connector_id << ", "
              "type=" << connector->connector_type << ", "
              "type_id=" << connector->connector_type_id);
    /*check for existing configuration*/
    if ( connector->encoder_id != 0)
    {
        encoder = drmModeGetEncoder(m_fdDev, connector->encoder_id);
        if (encoder == NULL)
        {
            LOG_WARNING("Gal2dGraphicSystem", "could not get encoder for connector id "
                    << connector->connector_id << ", "
                    "type= " << connector->connector_type << ", "
                    "type_id= " << connector->connector_type_id <<" found. errno: "<< errno );
            return -1;
        }

        if (encoder->crtc_id != 0)
        {
            /*the current encoder is active
             *means we have existing drm configuration for this connector
             *but still we need to check if dual plain crtc is used if requested*/
            if (preferred_crtc >= 0)
            {
                if ( (encoder->crtc_id == m_dualplain_crtcs[0]) ||
                      (encoder->crtc_id == m_dualplain_crtcs[1]))
                {
                    /*if preferred crtc is not that is used
                     * we need to ensure we will use this once again
                     * for the next connector which requires dualplain crtc*/
                    if (preferred_crtc != encoder->crtc_id)
                    {
                        m_dualplain_crtcs[1] = m_dualplain_crtcs[0];
                        m_dualplain_crtcs[0] = preferred_crtc;
                    }
                }
                else
                {
                    /*give this crtc back to available ctrc list
                     * we will reconnect this connector with non dualplain crtc*/
                    m_availableCrtcs |= 1 << encoder->crtc_id;
                    encoder->crtc_id = preferred_crtc;
                }
            }
            found_crtc = encoder->crtc_id;
        }

        drmModeFreeEncoder(encoder);
    }

    if ( found_crtc == 0)
    {
        /*no drm configuration exists*/

        /*iterate over the list of supported encoders*/
        for (ii = 0; ii < connector->count_encoders; ii++)
        {
            encoder = drmModeGetEncoder(m_fdDev, connector->encoders[ii]);
            if (encoder == NULL)
            {
                LOG_WARNING("Gal2dGraphicSystem", "could not get encoder for connector id "
                            << connector->connector_id << ", "
                            "type= " << connector->connector_type << ", "
                            "type_id= " << connector->connector_type_id <<" found. errno: "<< errno );
                return -1;
            }

            if (encoder->crtc_id == 0)
            {
                /*encoder is free to use*/
                /*remember crtcs which are free*/
                uint32_t free_crtcs = m_availableCrtcs & (m_availableCrtcs ^ m_usedCrtcs);

                /*iterate over all crtcs*/
                for (k = 0; (k < m_crtcsNum) && (free_crtcs > 0); k++)
                {
                    drmModeCrtc* crtc = NULL;
                    int curr_crtc_id = 0 ;

                    /*check if crtc is compatible with encoder*/
                    if ( !((1 << k) & encoder->possible_crtcs) )
                        continue;

                    crtc = (drmModeCrtc*)m_crtcs[k];
                    if (crtc)
                    {
                        curr_crtc_id = crtc->crtc_id;

                        /*check if crtc is free to be used*/
                        if (free_crtcs & (1 << curr_crtc_id))
                        {
                            encoder->crtc_id = curr_crtc_id;
                            found_crtc = curr_crtc_id;
                            break;
                        }
                    }
                }
            }

            drmModeFreeEncoder(encoder);

            /*encoder found, stop the loop*/
            if (found_crtc != 0)
            {
                break;
            }
        }
    }

    /* we still have no (free)crtc for this encoder so exit with error
     *  ==> no crtc assignment for this connector */
    if ( found_crtc == 0)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "No free CRTC for encoder found,"
                    " connector type=" << connector->connector_type<<
                    " type_id= " << connector->connector_type_id);
        return -1;
    }

    DrmOutput* output = (DrmOutput*)malloc(sizeof *output);
    if (output == NULL)
    {
        LOG_ERROR("Gal2dGraphicSystem",
                "could not allocate memory for output")
        return -1;
    }
    memset(output, 0x00, sizeof *output);

    output->isScreenshot = false;
    output->windowSystem = m_baseWindowSystem;
    output->graphicSystem = this;
    output->fdDev       = m_fdDev;
    output->crtcID      = found_crtc;
    output->connectorID = connector->connector_id;
    output->connector_type =  connector->connector_type,
    output->connector_type_id = connector->connector_type_id;
    output->screen_alpha = 1.0f;
    output->repaintWindowPeriodMS = mConfiguration->getRepaintWindowPeriodMS();

    m_usedCrtcs |= (1 << found_crtc);

    wl_list_init(&output->modeList);

    output->orgCrtc = drmModeGetCrtc(m_fdDev, output->crtcID);
    for (ii = (connector->count_modes -1); ii >= 0 ; --ii)
    {
        if (drmOutputAddMode(output, &connector->modes[ii]))
          goto err_free;
    }

    wl_list_for_each(mode, &output->modeList, link)
    {
       drmMode = wl_container_of(output->modeList.next, drmMode, link);
       ptr = output->modeList.next;

       if ((drmMode->modeInfo.hdisplay == output->orgCrtc->mode.hdisplay) &&
           (drmMode->modeInfo.vdisplay == output->orgCrtc->mode.vdisplay) &&
           (drmMode->modeInfo.hsync_start == output->orgCrtc->mode.hsync_start) &&
           (drmMode->modeInfo.vsync_start == output->orgCrtc->mode.vsync_start))
       {
           LOG_DEBUG("Gal2dGraphicSystem",
                             "CreateOutputForConnector - Mode retrieval -"
                             << "hdisplay: " << drmMode->modeInfo.hdisplay
                             << "vdisplay: " << drmMode->modeInfo.vdisplay
                             << "hsync_start" << drmMode->modeInfo.hsync_start
                             << "vsync_start" << drmMode->modeInfo.vsync_start);
           break;
       }
       else
       {
           output->modeList.next = ptr->next;
           continue;
       }
    }

    LOG_INFO("Gal2dGraphicSystem",
             "screen=" << screen_id << ", "
             "currentMode->refresh_ns=" << drmMode->refresh_ns);

    output->currentMode = drmMode;
    drmMode->flags = WL_OUTPUT_MODE_CURRENT | WL_OUTPUT_MODE_PREFERRED;

    screen_id = (uint)connectorDispID;
    output->screenID = screen_id;
    output->dstNumSurfaces = mConfiguration->getScreenBufferCnt(screen_id);
    output->dither = mConfiguration->getScreenDithering(screen_id);
    output->depth = mConfiguration->getScreenColorDepth(screen_id);

    if (output->depth == 16)
        output->surfaceFormat = gcvSURF_R5G6B5;
    else if (output->depth == 32)
        output->surfaceFormat = gcvSURF_X8R8G8B8;
    else
        output->surfaceFormat = gcvSURF_R5G6B5;

    output->dstSurfaces = new gcoSURF[output->dstNumSurfaces];
    output->buffer = new struct drm_buffer[output->dstNumSurfaces];

    for (output->currentIndex = 0;
         output->currentIndex < output->dstNumSurfaces;
         output->currentIndex++)
    {
        /*align the width of the buffer as required by the gpu hardware*/
        gctUINT xalignment = 0;
        status =  gcoSURF_GetAlignment(gcvSURF_BITMAP,output->surfaceFormat, NULL, &xalignment, NULL);
        if (gcvSTATUS_OK != status)
        {
            LOG_ERROR("Gal2dGraphicSystem","error by gcoSURF_GetAlignment"
                    "required to align the drm buffer size, status: "
                    << status);
            goto err_free;
        }
        output->buffer[output->currentIndex].width =
                gcmALIGN_NP2(output->currentMode->width,xalignment);
        output->buffer[output->currentIndex].height = output->currentMode->height;

        ret = createBuffer( &output->buffer[output->currentIndex],output->depth);
        if (ret < 0)
        {
            LOG_ERROR("Gal2dGraphicSystem",
                        "Failed to create output buffer="
                        << output->currentIndex << ", "
                        "target CRTC ID=" << output->crtcID << ", "
                        "connector ID=" << output->connectorID << ", "
                        "screen ID=" << output->screenID);
            goto err_free;
        }
        status = gcoSURF_ConstructWrapper(m_gcoHal, &output->dstSurfaces[output->currentIndex]);
        if (gcvSTATUS_OK != status)
        {
            LOG_ERROR("Gal2dGraphicSystem",
                        "Failed to create gcoSURF wrapper, "
                        "target CRTC ID=" << output->crtcID << ", "
                        "connector ID=" << output->connectorID << ", "
                        "screen ID=" << output->screenID << ", "
                        "gcoSURF="
                        << output->dstSurfaces[output->currentIndex]);
            goto err_free;
        }
        status = gcoSURF_SetBuffer(output->dstSurfaces[output->currentIndex], gcvSURF_BITMAP,
                                   output->surfaceFormat, ~0U,
                                   output->buffer[output->currentIndex].pixels,
                                   ~0U);
        if (gcvSTATUS_OK != status)
        {
            LOG_ERROR("Gal2dGraphicSystem",
                        "Failed to set gcoSURF Buffer, "
                        "target CRTC ID=" << output->crtcID << ", "
                        "connector ID=" << output->connectorID << ", "
                        "screen ID=" << output->screenID << ", "
                        "gcoSURF="
                        << output->dstSurfaces[output->currentIndex]);
            goto err_free;
        }

        status = gcoSURF_SetWindow(output->dstSurfaces[output->currentIndex],
                                   0,
                                   0,
                                   output->buffer[output->currentIndex].width,
                                   output->buffer[output->currentIndex].height);

        if (gcvSTATUS_OK != status)
        {
            LOG_ERROR("Gal2dGraphicSystem", "Failed to set window, "
                        "target dstNumSurfaces=" << output->dstNumSurfaces << ", "
                        "target surface format=" << output->surfaceFormat << ", "
                        "target CRTC ID=" << output->crtcID << ", "
                        "connector ID=" << output->connectorID << ", "
                        "screen ID=" << output->screenID << ", "
                        "gcoSURF="
                        << output->dstSurfaces[output->currentIndex] << ", "
                        "width="
                        << output->buffer[output->currentIndex].width << ", "
                        "height="
                        << output->buffer[output->currentIndex].height);
            goto err_free;
        }

        LOG_DEBUG("Gal2dGraphicSystem",
                  "Buffer Created - "
                  "target dstNumSurfaces=" << output->dstNumSurfaces << ", "
                  "target surface format=" << output->surfaceFormat << ", "
                  "target CRTC ID=" << output->crtcID << ", "
                  "connector ID=" << output->connectorID << ", "
                  "screen ID=" << screen_id << ", "
                  "gcoSURF="
                  << output->dstSurfaces[output->currentIndex] << ", "
                  "width="
                  << output->buffer[output->currentIndex].width << ", "
                  "height=" << output->buffer[output->currentIndex].height);
    }

    if (drmModeSetCrtc(m_fdDev, output->crtcID,
                            output->buffer[output->dstNumSurfaces-1].fb, 0, 0,
                            &output->connectorID, 1,
                            &output->currentMode->modeInfo))
    {
        int drmModeErrno = errno;
        char message[250];
        int sem_value = 666;
        if (m_drm_master_sem)
        {
            sem_getvalue(m_drm_master_sem, &sem_value);
        }
        sprintf(message,"LAYERMANAGER: error by drmModeSetCrtc,errno: %d,crtc_id: %d,drm_fb: %d conn_id: %d "
                " origmod:%dx%d hss:%d vss:%d"
                " curmode:%dx%d hss:%d vss:%d"
                " ,drm_sem: %p,value: %d",
                        drmModeErrno,output->crtcID,
                        output->buffer[output->dstNumSurfaces-1].fb,output->connectorID,
                        output->orgCrtc->mode.hdisplay,
                        output->orgCrtc->mode.vdisplay,
                        output->orgCrtc->mode.hsync_start,
                        output->orgCrtc->mode.vsync_start,
                        drmMode->modeInfo.hdisplay,
                        drmMode->modeInfo.vdisplay,
                        drmMode->modeInfo.hsync_start,
                        drmMode->modeInfo.vsync_start,
                        (void*)m_drm_master_sem,
                        sem_value);

        log_to_errmem(message);

        LOG_ERROR("Gal2dGraphicSystem",
                    "Failed to set mode, "
                    "Target dstNumSurfaces=" << output->dstNumSurfaces << ", "
                    "Target surface format="
                    << output->surfaceFormat << ", "
                    "Target CRTC ID=" << output->crtcID << ", "
                    "Connector ID=" << output->connectorID << ", "
                    "Screen ID=" << output->screenID << ", "
                    "gcoSURF=" << output->dstSurfaces[0] << ", "
                    "width=" << output->buffer[0].width << ", "
                    "height=" << output->buffer[0].height);
        return -1;
    }

    output->currentIndex = 0;
    m_currentOutput = output;

    wl_list_insert(m_outputList.prev, &output->link);

    createOutputForScreenshot(output);

    setTargetSurface(output->dstSurfaces[output->currentIndex]);

    output->propDPMS = getDRMProperty(m_fdDev, connector, (char *)"DPMS");

    output->clrprop.csc.enable = true;
    output->clrprop.csc.type = DRM_IMX_CSC_RGB2RGB;
    output->clrprop.csc.crtc_id = output->crtcID;
    csc_set_default_value(&output->clrprop);
    csc_update_crtc_id(INVALID_VALUE);

    /*set the default CSC properties for the screen if available*/
    struct ilmCSCProperties* pclrProp;
    pclrProp = mConfiguration->getCSCValues(output->screenID);
    if(NULL != pclrProp)
    {
       setCSC(output->screenID, pclrProp);
    }

    LOG_DEBUG("Gal2dGraphicSystem", "Exited");
    return 0;

err_free:
    drmModeFreeCrtc(output->orgCrtc);
    m_usedCrtcs &= ~(1 << output->crtcID);

    free(output);
    LOG_WARNING("Gal2dGraphicSystem",
                "Exited, connector ID="
                << connector->connector_id);
    return -1;
}


int Gal2dGraphicSystem::createBuffer(struct drm_buffer* buffer,int bpp)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called, buffer address=" << buffer);

    struct drm_mode_create_dumb cre_req;
    struct drm_mode_destroy_dumb dest_req;
    struct drm_mode_map_dumb map_req;
    unsigned int depth = 0;
    int errorNo = 0;
    int ret;
    /*check input parameters*/
    if ( (buffer->width <= 0) || (buffer->height <= 0))
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Invalid dimensions for display "
                    "buffer: (w="<< buffer->width << ", "
                    << " h="<< buffer->height << ")");
        return -1;
    }
    if (bpp == 32)
    {
        depth = 24;
    }
    else if (bpp == 16)
    {
        depth = 16;
    }
    else
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Invalid depth for display buffer, depth=" << depth );
        return -1;
    }
    /*set create dumb structure */
    memset(&cre_req, 0, sizeof(cre_req));

    cre_req.width = buffer->width;
    cre_req.height = buffer->height;
    cre_req.bpp = bpp;

    errno = 0;

    ret = drmIoctl(m_fdDev, DRM_IOCTL_MODE_CREATE_DUMB, &cre_req);
    errorNo = errno;
    if (ret < 0)
    {
        char errbuf[BUFSIZ];
        strerror_r(errorNo, errbuf, BUFSIZ);

        LOG_WARNING("Gal2dGraphicSystem",
                    "Cannot create dumb buffer - Return value="<< ret << ", "
                    "error=" << errbuf);
        return -ret;
    }

    buffer->stride = cre_req.pitch;
    buffer->size = cre_req.size;
    buffer->handle = cre_req.handle;
    errno = 0;
    /* create framebuffer object for the dumb-buffer */
    ret = drmModeAddFB(m_fdDev,
                    buffer->width,
                    buffer->height,
                    depth,cre_req.bpp ,
                    buffer->stride,
                    buffer->handle,
                    &buffer->fb);
    errorNo = errno;
    if (ret)
    {
        char errbuf[BUFSIZ];
        strerror_r(errorNo, errbuf, BUFSIZ);

        LOG_WARNING("Gal2dGraphicSystem",
                    "Cannot create framebuffer: Return value=" << ret << ", "
                    "error=" << errbuf);

        memset(&dest_req, 0, sizeof(dest_req));
        dest_req.handle = buffer->handle;
        drmIoctl(m_fdDev, DRM_IOCTL_MODE_DESTROY_DUMB, &dest_req);
        return -ret;

    }

    /* prepare buffer for memory mapping */
    memset(&map_req, 0, sizeof(map_req));
    map_req.handle = buffer->handle;
    errno = 0;
    ret = drmIoctl(m_fdDev, DRM_IOCTL_MODE_MAP_DUMB, &map_req);
    errorNo = errno;
    if (ret)
    {
        char errbuf[BUFSIZ];
        strerror_r(errorNo, errbuf, BUFSIZ);

        LOG_WARNING("Gal2dGraphicSystem",
                    "Cannot map dumb buffer: Return value=" << ret << ", "
                    "error=" << errbuf);

        drmModeRmFB(m_fdDev, buffer->fb);
        memset(&dest_req, 0, sizeof(dest_req));
        dest_req.handle = buffer->handle;
        drmIoctl(m_fdDev, DRM_IOCTL_MODE_DESTROY_DUMB, &dest_req);
        return -ret;
    }

    errno = 0;

    /* perform actual memory mapping */
    buffer->pixels = mmap(0, buffer->size,
                          PROT_READ | PROT_WRITE,
                          MAP_SHARED,
                          m_fdDev,
                          map_req.offset);
    errorNo = errno;

    if (buffer->pixels == MAP_FAILED)
    {
        char errbuf[BUFSIZ];
        strerror_r(errorNo, errbuf, BUFSIZ);

        LOG_WARNING("Gal2dGraphicSystem",
                    "Cannot mmap dumb buffer, Error=" << errbuf);

        drmModeRmFB(m_fdDev, buffer->fb);
        memset(&dest_req, 0, sizeof(dest_req));
        dest_req.handle = buffer->handle;
        drmIoctl(m_fdDev, DRM_IOCTL_MODE_DESTROY_DUMB, &dest_req);
        return -errno;
    }

    /*SWGIII-26561: clearing of the drm buffer on userspace is removed as it is
     *also done at kernel space. We are now depending on the memset in kernel space
     *(file: drivers/base/dma-coherent.c; api: dma_alloc_from_coherent)
     *This is done to optimize start-up time of layermanager*/

    return 0;
}

static unsigned int calcRefreshNs(drmModeModeInfo* info)
{
    uint32_t millihz;
    uint32_t refresh_ns;

    /* Calculate higher precision (mHz) refresh rate */
    millihz = (info->clock * 1000000LL / info->htotal +
               info->vtotal / 2) / info->vtotal;

    if (info->flags & DRM_MODE_FLAG_INTERLACE)
        millihz *= 2;
    if (info->flags & DRM_MODE_FLAG_DBLSCAN)
        millihz /= 2;
    if (info->vscan > 1)
        millihz /= info->vscan;

    /*millihz to ns*/
    refresh_ns = (1000000000000LL / millihz);

    return refresh_ns;
}

int Gal2dGraphicSystem::drmOutputAddMode(struct DrmOutput* output, drmModeModeInfo* info)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called, drm output address=" << output);

    struct DrmMode* mode;
    uint32_t refresh_millihz;

    mode = (struct DrmMode*)malloc(sizeof *mode);
    if (mode == NULL)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Out of memory in drmOutputAddMode, "
                    "drm info mode: " << info->hdisplay <<"x"<< info->vdisplay
                    );
        return -1;
    }

    mode->flags    = 0;
    mode->width    = info->hdisplay;
    mode->height   = info->vdisplay;
    mode->refresh  = info->vrefresh;
    mode->modeInfo = *info;
    mode->refresh_ns = calcRefreshNs(info);

    wl_list_insert(output->modeList.prev, &mode->link);

    return 0;
}

void Gal2dGraphicSystem::updateScreenList(LmScreenList& screenList)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called");

    LmScreenListIterator iter = screenList.begin();
    LmScreenListIterator iterEnd = screenList.end();
    for (; iter != iterEnd; ++iter)
    {
        delete (*iter);
    }
    screenList.clear();
    struct DrmOutput* output = NULL;
    wl_list_for_each(output, &m_outputList, link)
    {
        output->screenID = getScreenIdFromConnectorType(output->connector_type,
                                                        output->connector_type_id);
        LmScreen* lmScreen = new LmScreen(output->screenID, "");
        screenList.push_back(lmScreen);
    }

    screenList.sort(compare_screen_ids);
    iter = screenList.begin();
    iterEnd = screenList.end();
    for (; iter != iterEnd; ++iter)
    {
        LOG_INFO("Gal2dGraphicSystem", "screen list IDs: "<<(*iter)->getID());
    }
}
unsigned int* Gal2dGraphicSystem::getScreenIDs(unsigned int *nScreens)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called");

    *nScreens = wl_list_length(&m_outputList);
    unsigned int *ids = new unsigned int[*nScreens];
    int i = 0;
    DrmOutput *output = NULL;
    wl_list_for_each(output, &m_outputList, link){
        ids[i]= output->screenID;
        i++;
    }

    return ids;
}

unsigned int* Gal2dGraphicSystem::getScreenResolution(uint screenID)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called, screen ID=" << screenID);

    uint * resolution = new uint[2];
    DrmOutput *output = NULL;
    resolution[0] =-1;
    resolution[1] =-1;

    wl_list_for_each(output, &m_outputList, link)
    {
        if (output->screenID != screenID)
        {
                continue;
        }
        resolution[0] = output->currentMode->width;
        resolution[1] = output->currentMode->height;
    }
    return resolution;
}

void Gal2dGraphicSystem::getScreenResolution(int screenID, int &width, int &height)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called, screen ID=" << screenID);

    DrmOutput *output = NULL;
    width = -1;
    height = -1;

    wl_list_for_each(output, &m_outputList, link)
    {
        if ((int)output->screenID != screenID)
        {
                continue;
        }
        width =  (int)output->currentMode->width;
        height = (int)output->currentMode->height;
    }
}

static void ilm_wl_drm_authenticate(struct wl_client *client,
		struct wl_resource *resource, uint32_t id)
{
	int* pDrmFd = (int*)resource->data;

	if( drmAuthMagic(*pDrmFd, id) < 0)
		wl_resource_post_error(resource,
								ILM_WL_DRM_ERROR_AUTHENTICATE_FAIL,
								"authenicate failed");
	else
		ilm_wl_drm_send_authenticated(resource);
}

static const struct ilm_wl_drm_interface ilm_wl_drm_implementation = {
		ilm_wl_drm_authenticate
};

static void bind_ilm_wl_drm(struct wl_client *client, void *data,
		uint32_t version, uint32_t id)
{
	struct wl_resource *resource;

	resource = wl_resource_create(client, &ilm_wl_drm_interface,
				      version, id);
	if (resource == NULL) {
		wl_client_post_no_memory(client);
		return;
	}

	wl_resource_set_implementation(resource, &ilm_wl_drm_implementation,
				       data, NULL);
}

bool Gal2dGraphicSystem::init(void* display, void* window)
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, display address=" << display << ", "
              "window address=" << window);

    gceSTATUS status = gcvSTATUS_OK;

    m_nativeWindow = (void*)window;

    setenv("FB_MULTI_BUFFER", "0", 1);

    status = gcoOS_GetDisplay(&m_nativeDisplay, m_nativeWindow);
    if ((gcvSTATUS_OK != status) || (NULL == m_nativeDisplay))
    {
        LOG_WARNING("Gal2dGraphicSystem", "Failed to get HAL display");
        return false;
    }
    LOG_DEBUG("Gal2dGraphicSystem", "Got HAL display=" << m_nativeDisplay);

    //needed to register the wl_viv protocol
    status = gcoOS_InitLocalDisplayInfo(m_nativeDisplay, &m_localDisplayInfo);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to initialize localdisplayinfo, "
                    "display address=" << display << ", "
                    "window address=" << window);
        return false;
    }

    wl_list_init(&m_outputList);
    wl_list_init(&m_screenshotList);

    if (!initGal2d(m_windowWidth, m_windowHeight))
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Initialization failed, "
                    "display address=" << display << ", "
                    "window address=" << window);
        return false;
    }

    if (!initializeDrmFD())
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Cannot initialize drm device, "
                    "display address=" << display << ", "
                    "window address=" << window);
        return false;
    }

    if (!initializeSystem())
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Initialization of system failed, "
                    "display address=" << display << ", "
                    "window address=" << window);
        return false;
    }

    if(!wl_global_create((struct wl_display *)window,
                          &ilm_wl_drm_interface, 1,
                          (void*)&m_fdDev, bind_ilm_wl_drm))
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "ilm_wl_drm potocol init failed");
        return false;
    }

    LOG_DEBUG("Gal2dGraphicSystem",
              "Initialization successful, "
                    "display address=" << display << ", "
                    "window address=" << window);

    sem_init(&m_workerSem, 0, 1);
    return true;
}

void Gal2dGraphicSystem::clearBackground()
{
    LOG_DEBUG("Gal2dGraphicSystem","Called");

    gctINT stride = 0;
    gctUINT width = 0;
    gctUINT height = 0;
    gcsRECT rect = {0, 0, 0, 0};
    gceSTATUS status = gcvSTATUS_OK;

    status = gcoSURF_GetAlignedSize(m_currentOutput->dstSurfaces[m_currentOutput->currentIndex], &width, &height, &stride);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get aligned size, "
                    "Target dstNumSurfaces=" << m_currentOutput->dstNumSurfaces << ", "
                    "Target surface format=" << m_currentOutput->surfaceFormat << ", "
                    "Target CRTC ID=" << m_currentOutput->crtcID << ", "
                    "Connector ID=" << m_currentOutput->connectorID << ", "
                    "Screen ID=" << m_currentOutput->screenID << ", "
                    "gcoSURF="
                    << m_currentOutput->dstSurfaces[m_currentOutput->currentIndex] << ", "
                    "width="
                    << m_currentOutput->buffer[m_currentOutput->currentIndex].width << ", "
                    "height="
                    << m_currentOutput->buffer[m_currentOutput->currentIndex].height);
        return;
    }

    rect.right = width;
    rect.bottom = height;
    status = gco2D_SetSource(m_gco2dEngine, &rect);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to setSource, "
                    "Target dstNumSurfaces=" << m_currentOutput->dstNumSurfaces << ", "
                    "Target surface format=" << m_currentOutput->surfaceFormat << ", "
                    "Target CRTC ID=" << m_currentOutput->crtcID << ", "
                    "Connector ID=" << m_currentOutput->connectorID << ", "
                    "Screen ID=" << m_currentOutput->screenID << ", "
                    "gcoSURF="
                    << m_currentOutput->dstSurfaces[m_currentOutput->currentIndex] << ", "
                    "width="
                    << m_currentOutput->buffer[m_currentOutput->currentIndex].width << ", "
                    "height="
                    << m_currentOutput->buffer[m_currentOutput->currentIndex].height);
        return;
    }

    status = gco2D_SetClipping(m_gco2dEngine, &rect);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to set clipping area, "
                    "Target dstNumSurfaces=" << m_currentOutput->dstNumSurfaces << ", "
                    "Target surface format=" << m_currentOutput->surfaceFormat << ", "
                    "Target CRTC ID=" << m_currentOutput->crtcID << ", "
                    "Connector ID=" << m_currentOutput->connectorID << ", "
                    "Screen ID=" << m_currentOutput->screenID << ", "
                    "gcoSURF="
                    << m_currentOutput->dstSurfaces[m_currentOutput->currentIndex] << ", "
                    "width="
                    << m_currentOutput->buffer[m_currentOutput->currentIndex].width << ", "
                    "height="
                    << m_currentOutput->buffer[m_currentOutput->currentIndex].height);
        return;
    }

    status = gco2D_Clear(m_gco2dEngine, 1, &rect, 0xFF000000, 0xCC, 0xCC, m_currentOutput->surfaceFormat);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to clear, "
                    "Target dstNumSurfaces=" << m_currentOutput->dstNumSurfaces << ", "
                    "Target surface format=" << m_currentOutput->surfaceFormat << ", "
                    "Target CRTC ID=" << m_currentOutput->crtcID << ", "
                    "Connector ID=" << m_currentOutput->connectorID << ", "
                    "Screen ID=" << m_currentOutput->screenID << ", "
                    "gcoSURF="
                    << m_currentOutput->dstSurfaces[m_currentOutput->currentIndex] << ", "
                    "width="
                    << m_currentOutput->buffer[m_currentOutput->currentIndex].width << ", "
                    "height="
                    << m_currentOutput->buffer[m_currentOutput->currentIndex].height << ", "
		    "status=" << status);
        return;
    }
    LOG_DEBUG("Gal2dGraphicSystem",
              "Target dstNumSurfaces=" << m_currentOutput->dstNumSurfaces << ", "
              "Target surface format=" << m_currentOutput->surfaceFormat << ", "
              "Target CRTC ID=" << m_currentOutput->crtcID << ", "
              "Connector ID=" << m_currentOutput->connectorID << ", "
              "Screen ID=" << m_currentOutput->screenID << ", "
              "gcoSURF="
              << m_currentOutput->dstSurfaces[m_currentOutput->currentIndex] << ", "
              "width="
              << m_currentOutput->buffer[m_currentOutput->currentIndex].width << ", "
              "height="
              << m_currentOutput->buffer[m_currentOutput->currentIndex].height);
}

void Gal2dGraphicSystem::flushAndCommit()
{
    gceSTATUS status = gcvSTATUS_OK;

    LOG_DEBUG("Gal2dGraphicSystem", "Called");

    status = gco2D_Flush(m_gco2dEngine);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to flush 2d area, "
                    "CRTC ID=" << m_currentOutput->crtcID << ", "
                    "connector ID=" << m_currentOutput->connectorID << ", "
                    "screen ID=" << m_currentOutput->screenID << ", "
                    "gcoSURF="
                    << m_currentOutput->currentTargetSurface);
    }
    status = gcoHAL_Commit(m_gcoHal, gcvTRUE);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to commit(sync) area, "
                    "CRTC ID=" << m_currentOutput->crtcID << ", "
                    "connector ID=" << m_currentOutput->connectorID << ", "
                    "screen ID=" << m_currentOutput->screenID << ", "
                    "gcoSURF="
                    << m_currentOutput->currentTargetSurface);
    }

}

bool Gal2dGraphicSystem::destroyVncbuffer(int screenID,void *p_vncsurfptr)
{
    screenID = screenID;
    gcoSURF_Destroy((gcoSURF)p_vncsurfptr);
    return true;
}

bool Gal2dGraphicSystem::dumpFramebuffertovncsurf(int screenID)
{
    WaylandBaseWindowSystem* windowSystem = dynamic_cast<WaylandBaseWindowSystem*>(m_baseWindowSystem);
     void    *p_vncbuffaddr = NULL;
     void    *p_vncsurfptr = NULL;
    int ret_val = false;

    LOG_DEBUG("Gal2dGraphicSystem", "called screenID="<<screenID);

     windowSystem->getVncDisplayConnectionState(screenID,
                                         &p_vncbuffaddr,&p_vncsurfptr);
      /*
        * if vnc display connection state is VNC_BUFFER_FREE,
        * then will get the vnc surf pointer 
        * else vncsurf pointer will be NULL
        */
     if( NULL != p_vncsurfptr )
    {
         LOG_DEBUG("Gal2dGraphicSystem", "dumping screenID="<<screenID);
         windowSystem->NotifyVnc_BeforeFramebufferUpdate(screenID);
         dumpSurfacetovncsurf(screenID,p_vncsurfptr);
         windowSystem->NotifyVnc_FramebufferUpdated(screenID);
         ret_val = true;
     }
    return ret_val;
}

bool Gal2dGraphicSystem::createVncbuffer( int screenID,int bpp,void **pp_vncsurfptr,void **pp_buffaddr)
{
    unsigned int* screenResolution;
    gceSURF_FORMAT format;
    bool status = true;
    gctUINT32 phyAddr;
    gctPOINTER logAddr;
    gctINT stride = 0;
    gctUINT width = 0;
    gctUINT height = 0;
    gcsRECT rect = {0, 0, 0, 0};

    LOG_DEBUG("Gal2dGraphicSystem", "called bpp="<<bpp<<", "
                        "screenID="<<screenID);

    *pp_vncsurfptr = NULL;
    *pp_buffaddr = NULL;
    screenResolution = getScreenResolution(screenID);

    if (16 == bpp)
    {
      format = gcvSURF_A4B4G4R4;
    }
    else if(32 == bpp)
    {
        format = gcvSURF_A8B8G8R8;
    }
    /* by default, configure the target screen formats */
    else if(-1 == bpp)
    {
        status = gcoSURF_GetFormat(m_currentOutput->currentTargetSurface, gcvNULL, &format);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to get format, "
                        "status=" << status);
            goto err;
        }
    }
    else
    {
        LOG_ERROR("Gal2dGraphicSystem",
                  "Not supported BPP configured="<<(gctUINT)bpp);

        goto err;

    }

        status = gcoSURF_Construct(m_gcoHal,
                               screenResolution[0],
                               screenResolution[1],
                               1,
                               gcvSURF_BITMAP,
                               format,
                               gcvPOOL_DEFAULT,
                               (gcoSURF*)pp_vncsurfptr );

    if (status != gcvSTATUS_OK)
    {
        LOG_ERROR("Gal2dGraphicSystem",
                  "Failed to construct VNC Gco surface");

        goto err;
    }

    status = gcoSURF_Lock((gcoSURF)*pp_vncsurfptr,
                          &phyAddr,
                          (gctPOINTER*)&logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to lock surface, "
                    "status=" << status);
        goto err;
    }

    LOG_DEBUG("Gal2dGraphicSystem", "width="<<screenResolution[0]<<", "
                        "height="<<screenResolution[1]<<", "
                        "surf_ptr="<<*pp_vncsurfptr <<", "
                        "surf_buff_addr="<<logAddr);

    status = gcoSURF_GetAlignedSize((gcoSURF)*pp_vncsurfptr, &width, &height, &stride);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get aligned size of vnc gcosurf");
    }
    status = gco2D_SetTargetEx(m_gco2dEngine, phyAddr, stride, gcvSURF_0_DEGREE, width, height);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to SetTargetEx of vnc gcosurf ");
    }

    rect.right = width;
    rect.bottom = height;
    status = gco2D_SetSource(m_gco2dEngine, &rect);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to setSource of vnc gcosurf ");
    }

    status = gco2D_SetClipping(m_gco2dEngine, &rect);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to set clipping area of vnc gcosurf ");
    }

    status = gco2D_Clear(m_gco2dEngine, 1, &rect, 0xFF000000, 0xCC, 0xCC, format);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to clear of vnc gcosurf ");
    }

    status = gcoSURF_Unlock((gcoSURF)*pp_vncsurfptr,
                                                (gctPOINTER)logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to unlock surface, "
                    "gcoSURF=" << *pp_vncsurfptr);
    }

    *pp_buffaddr = (void*)logAddr;
    return status;

err:
    if(NULL != *pp_vncsurfptr)
    {
        gcoSURF_Destroy((gcoSURF)*pp_vncsurfptr);
        *pp_buffaddr =NULL;
        *pp_vncsurfptr = NULL;
    }
    return false;    
}

void* Gal2dGraphicSystem::dumpSurfacetovncsurf( unsigned int screenID,void *p_vncsurfptr)
{
    gceSTATUS status = gcvSTATUS_OK;
    gctINT stride;
    gctUINT width;
    gctUINT height;
    gctINT vnc_stride;
    gctUINT vnc_width;
    gctUINT vnc_height;
    gceSURF_FORMAT format;
    gctUINT32 phyAddr;
    gctPOINTER logAddr;
    gctUINT32 vnc_phyAddr;
    gctPOINTER vnc_logAddr;
   gcsRECT srcRect = {0, 0, 0, 0};
    gcsRECT dstRect = {0, 0, 0, 0};
    gcsRECT clipRect = {0, 0, 0, 0};
    gcoSURF vncsurf;
    struct DrmOutput* output = NULL;
    gcoSURF gcotargetsurf;
    gceTILING tiling = gcvINVALIDTILED;

    LOG_DEBUG("Gal2dGraphicSystem" ," called");

    vncsurf  = (gcoSURF)p_vncsurfptr;
    if( m_currentOutput->screenID == screenID)
    {
        output = m_currentOutput;
    }
    else
    {
        wl_list_for_each(output, &m_outputList, link)
        {
            if (output->screenID != screenID)
            {
                continue;
            }
            break;
        }
    }
    if(output == NULL)
    {
        LOG_ERROR("Gal2dGraphicSystem",
                "Failed to get the outut for given screenID = "<< screenID);
        return NULL;
    }

   gcotargetsurf = output->currentTargetSurface;

    status = gcoSURF_GetFormat(gcotargetsurf, gcvNULL, &format);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get format, "
                    "status=" << status);
        goto err_surf;
    }

    status = gcoSURF_GetAlignedSize(gcotargetsurf, &width, &height, &stride);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get aligned size, "
                    "status=" << status);
        goto err_surf;
    }

    status = gcoSURF_Lock(gcotargetsurf,
                          &phyAddr,
                          (gctPOINTER*)&logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to lock surface, "
                    "status=" << status);
        goto err_surf;
    }

    status = gcoSURF_GetTiling(gcotargetsurf, &tiling);

    if(gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                       "Failed to get tiling info, "
                        "status=" << status);
        gcoSURF_Unlock(gcotargetsurf, (gctPOINTER)logAddr);
        goto err_surf;
    }

    /*
     * Since the screen-framebuffer is a wrapped gcoSURF(created by gcoSURF_ConstructWrapper),
     * tiling info was not updated as gcvLINEAR as in the case of gcoSURF_Construct.
     * so here we will get the tiling info as gcvINVALIDTILED(0). In quad+, the
     * vnc dump is scattered if we set the tiling format as gcvINVALIDTILED.
     * So resetting it to gcvLINEAR*/
    if(tiling == gcvINVALIDTILED)
    {
            LOG_INFO("Gal2dGraphicSystem", "gcotargetsurf has Tiling as gcvINVALIDTILED, " <<
                     "resetting to gcvLINEAR");
            tiling = gcvLINEAR;
    }

    status = gco2D_SetGenericSource(m_gco2dEngine, &phyAddr, 1,
                        (gctUINT32_PTR)&stride, 1, tiling, format, gcvSURF_0_DEGREE,
                        width, height);

    if (gcvSTATUS_OK != status)
    {
                gcoSURF_Unlock(gcotargetsurf, (gctPOINTER)logAddr);
                goto err_surf;
    }

    srcRect.right  = width +  srcRect.left ;
    srcRect.bottom = height +  srcRect.right;

    status = gco2D_SetSource(m_gco2dEngine, &srcRect);
    if (gcvSTATUS_OK != status)
    {
        gcoSURF_Unlock(gcotargetsurf, (gctPOINTER)logAddr);
        goto err_surf;   
    }

    status = gcoSURF_Lock((gcoSURF)vncsurf,
                          &vnc_phyAddr,
                          (gctPOINTER*)&vnc_logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to lock surface, "
                    "status=" << status);
        goto err_surf;
    }

    status = gcoSURF_GetAlignedSize((gcoSURF)vncsurf, 
                                                        &vnc_width, &vnc_height, &vnc_stride);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get aligned size, "
                    "status=" << status);
        goto err_surf;
    }

    status = gcoSURF_GetFormat((gcoSURF)vncsurf, 
                                                    gcvNULL, &format);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get format, "
                    "status=" << status);
        goto err_surf;
    }

    if (format != gcvSURF_A4B4G4R4 && format != gcvSURF_A8B8G8R8)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Unexpected format, value=" << format << ", "
                    "instead of gcvSURF_A4R4G4B4 or gcvSURF_B8G8R8A8");
    }

    status = gco2D_SetTargetEx(m_gco2dEngine, vnc_phyAddr, vnc_stride, 
                                                gcvSURF_0_DEGREE, vnc_width, vnc_height);

    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to set target surface, "
                    "VNC gcoSURF" );
        goto err_surf;
    }

       clipRect.left   = 0;
       clipRect.top    = 0;
       clipRect.right  = (gctINT32)(( m_displayWidth -1 ) - clipRect.left);
       clipRect.bottom = (gctINT32)((m_displayHeight-1)- clipRect.top);

       status = gco2D_SetClipping(m_gco2dEngine, &clipRect);
       if (gcvSTATUS_OK != status)
       {
           LOG_WARNING("Gal2dGraphicSystem",
                       "Failed to set clipping"
                       " clipping area: "
                       "left=" << clipRect.left << ", "
                       "right=" << clipRect.right << ", "
                       "bottom=" << clipRect.bottom << ", "
                       "top=" << clipRect.top);
           goto err_surf;
       }

    dstRect.left   = (gctINT32)0;
    dstRect.top    = (gctINT32)0;
    dstRect.right  = (gctINT32)(vnc_width - dstRect.left);
    dstRect.bottom = (gctINT32)(vnc_height - dstRect.top);

    status = gco2D_Blit(m_gco2dEngine, 1, &dstRect, 0xCC, 0xAA, format);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to Blit, "<<
                    "status=" << status << ", "
                    "targetsurface"<<gcotargetsurf<<","
                    "Destination Area: "
                    "left=" << dstRect.left << ", "
                    "right=" << dstRect.right << ", "
                    "bottom=" << dstRect.bottom << ", "
                    "top=" << dstRect.top<<","
                    "tartget_format="<<output->surfaceFormat<<","
                    "vnc_format="<<format);
        goto err_surf;
    }

    status = gcoSURF_Unlock(gcotargetsurf,
                           (gctPOINTER)logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to unlock surface");
        goto err_surf;
    }

    status = gcoSURF_Unlock((gcoSURF)vncsurf,
                           (gctPOINTER)vnc_logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to unlock surface");
        goto err_surf;
    }

    flushAndCommit();
    return logAddr;

err_surf:
    return NULL;
}
void Gal2dGraphicSystem::swapBuffers()
{
   LOG_DEBUG("Gal2dGraphicSystem", "Called");

   flushAndCommit();
   if(m_currentOutput->dpms_current_state == DRM_MODE_DPMS_ON)
   {

       dumpFramebuffertovncsurf(m_currentOutput->screenID);

       if (drmModePageFlip(m_fdDev,
                            m_currentOutput->crtcID,
                            m_currentOutput->buffer[m_currentOutput->currentIndex].fb,
                            DRM_MODE_PAGE_FLIP_EVENT,
                            m_currentOutput) < 0)
        {
            LOG_ERROR("Gal2dGraphicSystem",
                        "Queueing pageflip failed, "
                        "errno=" << errno << ", "
                        "CRTC ID=" << m_currentOutput->crtcID << ", "
                        "connector ID=" << m_currentOutput->connectorID << ", "
                        "screen ID=" << m_currentOutput->screenID << ", "
                        "gcoSURF="
                        << m_currentOutput->currentTargetSurface);
            return;
        }
        m_currentOutput->pageFlipPending = true;

        clock_gettime(CLOCK_MONOTONIC, &m_currentOutput->time_pageflip);
    LOG_DEBUG("Gal2dGraphicSystem",
              "Exited: CRTC ID=" << m_currentOutput->crtcID << ", "
              "connector ID=" << m_currentOutput->connectorID << ", "
              "screen ID=" << m_currentOutput->screenID << ", "
              "gcoSURF="
              << m_currentOutput->currentTargetSurface);
   }
   LOG_DEBUG("Gal2dGraphicSystem", "Exited");
}

void Gal2dGraphicSystem::beginLayer(Layer* currentLayer)
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, current layer ID=" << currentLayer->getID() << ", "
              "layer=" << currentLayer);
    m_currentLayer = currentLayer;
}

void Gal2dGraphicSystem::endLayer()
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, layer ID=" << m_currentLayer->getID() << ", "
              "layer=" << m_currentLayer);
    m_currentLayer = NULL;
}

bool Gal2dGraphicSystem::synchronizedSurfacesDamaged(LayerList layers)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called, no. layers=" << layers.size());
    for (LayerListConstIterator layer = layers.begin(); layer != layers.end(); layer++)
    {
        if ((*layer)->getLayerType() == Hardware && layers.size() > 1)
        {
            // Damage in a hardware layer should not imply a redraw in other layers
            LOG_WARNING("GLESGraphicsystem",
                        "Called with layers "
                        "not in the same composition. layer=" << (*layer));
        }
        // Only check synchronized surfaces if the layer is visible
        if ((*layer)->visibility)
        {
            SurfaceList synchronizedSurfaces;
            SurfaceList surfaces = (*layer)->getAllSurfaces();

            // LOG_DEBUG("GLESGraphicsystem", "Checking surfaces");
            // Check if we got a synchronized surfaces, only if all synchronized surfaces are damaged, trigger a redraw
            for ( SurfaceListConstIterator surface = surfaces.begin(); surface != surfaces.end(); surface++ )
            {
                //LOG_DEBUG("GLESGraphicsystem", "Checking Surface for sychronization " << (*surface)->getID() );
                if ( ( (*surface)->synchronized || (*surface)->m_resizesync  ) && (*surface)->visibility)
                {
                    synchronizedSurfaces.push_back((*surface));
                }
            }
            // LOG_DEBUG("GLESGraphicsystem", "Checking synchronizedSurfaces");
            for ( SurfaceListConstIterator syncSurface = synchronizedSurfaces.begin(); syncSurface != synchronizedSurfaces.end(); syncSurface++ )
            {
                if ( !(*syncSurface)->damaged )
                {
                    LOG_DEBUG("Gal2dGraphicsystem",
                              "synchronized surface, ID="
                              << (*syncSurface)->getID() << ", "
                              "not damaged, skipping."
                              " layer=" << (*layer));
                    return false;
                } else
                {
                    LOG_DEBUG("Gal2dGraphicsystem","synchronized surface, ID="
                              << (*syncSurface)->getID() << ", "
                              "damaged, trigger composition. "
                              "layer=" << (*layer));
                }
            }
        }
    }
    return true;
}
// Reports whether a single layer is damaged/dirty
// Can not account for possible occlusion by other layers
bool Gal2dGraphicSystem::needsRedraw(Layer *layer)
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, layer ID=" << layer->getID() << ", "
              "layer=" << layer);

    if (layer->renderPropertyChanged)
    {
        return true;
    }

    if (layer->visibility && layer->opacity > 0.0)
    {
        SurfaceList surfaces = layer->getAllSurfaces();
        for(SurfaceListConstIterator currentS = surfaces.begin(); currentS != surfaces.end(); currentS++)
        {
            if ((*currentS)->renderPropertyChanged)
            {
                return true;
            }

            if ((*currentS)->hasNativeContent() && (*currentS)->damaged && (*currentS)->visibility && (*currentS)->opacity>0.0f)
            {
                return true;
            }
        }
    }

    return false;
}

bool Gal2dGraphicSystem::isSurfaceDirty(unsigned int surfID)
{
    Surface* surface = m_baseWindowSystem->m_pScene->getSurface(surfID);
    int layerId;
    Layer *layer;
    bool surfDirty = false;
    if (surface)
    {
        layerId = surface->getContainingLayerId();
        layer = m_baseWindowSystem->m_pScene->getLayer(layerId);

        if (NULL != layer)
        {
            if (layer->visibility && (layer->opacity > 0.0))
            {
                if ((surface->renderPropertyChanged)
                     || (   surface->hasNativeBuffer()
                         && surface->damaged
                         && surface->visibility
                         && (surface->opacity > 0.0f) ))
                {
                	surfDirty = true;
                }
            }
        }
    }
    return surfDirty;
}

bool Gal2dGraphicSystem::isScreenFrozen(unsigned int screenID)
{
	bool syncSurfDamaged = false;
    LayerList layers = m_baseWindowSystem->m_pScene->getCurrentRenderOrder(screenID);
    if (layers.size() > 0)
    {
    	syncSurfDamaged = synchronizedSurfacesDamaged(layers);
    }
    return (!syncSurfDamaged);
}
// Reports whether the passed in layers have visible damage or are otherwise
// dirty because render properties changed.
// Assumes that layers in the list belong to same composition. ie. damage to
// one layer affects the others.  A warning is logged if the assumption is wrong.
bool Gal2dGraphicSystem::needsRedraw(LayerList layers)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called, no. layers=" << layers.size());
    // TODO: Ignore damage from completely obscured surfaces

    for (LayerListConstIterator layer = layers.begin(); layer != layers.end(); layer++)
    {
        if ((*layer)->getLayerType() == Hardware && layers.size() > 1)
        {
            // Damage in a hardware layer should not imply a redraw in other layers
            LOG_WARNING("Gal2dGraphicSystem",
                        "needsRedraw() called with layers "
                        "not in the same composition. layer=" << (*layer));
        }

        if (needsRedraw(*layer))
        {
            return true;
        }
    }
    return false;
}

void Gal2dGraphicSystem::renderSWLayer(Layer *layer, bool clear)
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, layer ID=" << layer->getID() << ", "
              "layer=" << layer << ", clear=" << clear);

    beginLayer(layer);

    if ((layer->getLayerType() != Software_2D) &&
        (layer->getLayerType() != Software_2_5D))
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Called with a non-SW layer, "
                    "layer ID=" << layer->getID() << ", "
                    "layer=" << layer);
    }

#ifndef WL_OMIT_CLEAR_GB
    if (clear)
    {
        clearBackground();
    }
#else /* WL_OMIT_CLEAR_GB */
    clear = clear;
#endif /* WL_OMIT_CLEAR_GB */

    if ( layer->visibility && layer->opacity > 0.0 )
    {
        SurfaceList surfaces = layer->getAllSurfaces();
        for(std::list<Surface*>::const_iterator currentS = surfaces.begin(); currentS != surfaces.end(); currentS++)
        {
            if ((*currentS)->hasNativeContent() && (*currentS)->visibility && (*currentS)->opacity>0.0f)
            {
                renderSurface(*currentS);
            }
        }
    }

    endLayer();
}

// Decide if the "skip clear" optimization is legal to use, but not necessarily that
// it should be used.  That is determined in useSkipClear().
bool Gal2dGraphicSystem::canSkipClear()
{
    LOG_DEBUG("Gal2dGraphicSystem","Called");
    // No current cases where this approach cannot be used.
    return true;
}

// Decide if the "skip clear" optimization should be used.  This does not necessarily
// mean that the optimization is legal to use.  In this case, it is always legal, but
// not always wise.
bool Gal2dGraphicSystem::useSkipClear(LayerList layers)
{
    LOG_DEBUG("Gal2dGraphicSystem","Called, no. layers=" << layers.size());
    float surfaceArea, displayArea, threshold;
    SurfaceList surfaces;
    static int count = 0;
    count++;

    switch(m_optimizations[OPT_SKIP_CLEAR])
    {
    case OPT_MODE_FORCE_OFF:
        return false;
    case OPT_MODE_FORCE_ON:
        return true;
    case OPT_MODE_HEURISTIC:
        // Sample heuristic.  When large amounts of background are visible, it
        // may be better to disable this optimization and simply use a regular
        // glClear() call.  If very little, or no background is visible,
        // we can skip glClear() to save time.
        displayArea = m_displayWidth * m_displayHeight;
        surfaceArea = 0;
        threshold = .75; // 75% (Needs to be refined by experimentation)
        for(LayerListConstIterator layer = layers.begin(); layer != layers.end(); layer++)
        {
            surfaces = (*layer)->getAllSurfaces();

            if (surfaces.size() == 0) continue;
            if (!(*layer)->visibility || (*layer)->getOpacity() <= 0.0f) continue;

            for(SurfaceListConstIterator currentS = surfaces.begin(); currentS != surfaces.end(); currentS++)
            {
                if ((*currentS)->visibility == true && (*currentS)->getOpacity() > 0.0f)
                {
                    FloatRectangle surfDest = (*currentS)->getTargetDestinationRegion();
                    surfaceArea += surfDest.width * surfDest.height;
                }
            }
            break;
        }
        return (surfaceArea > threshold * displayArea);
    case OPT_MODE_TOGGLE:
        // Toggles optimization on and off to reveal bugs.  If flickering
        // appears, something is broken.  Optimizations should have no impact
        // on image appearance.  For debugging only.
        return (count % 20) > 10;
    default:
        LOG_WARNING("Gal2dGraphicSystem", "Bad SkipClear Optimization Mode");
        return false;
    }
}

// Composite the surfaces from all layers in the provided list.
// A traditional single-texture approach or a potentially faster multi-texture
// method may be used.
// 'clear' means that the framebuffer is dirty and needs to be initialized
// to known values.  Normally achieved through a glClear, it can also be satisfied
// by ensuring that each pixel receives at least one unblended draw.
void Gal2dGraphicSystem::renderSWLayers(LayerList layers, bool clear)
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, no. layers=" << layers.size() << ", clear=" << clear);

    bool optimizeClear = useSkipClear(layers) && canSkipClear();

    if (layers.size() == 0)
    {
        if (clear)
        {
            clearBackground();
        }
        return;
    }

#ifndef WL_OMIT_CLEAR_GB
    if (clear && !optimizeClear)
    {
        clearBackground();
        clear = false;
    }
#endif /* WL_OMIT_CLEAR_GB */

    SurfaceList opaquereg_surf_list =  getOpaqueRegSurfList(layers);

    //if multisrc is supported
    if (m_maxMultiSrc > 0)
    {
        LOG_INFO("Gal2dGraphicSystem",
                "multi src blit is available, "
                "trying to use it m_maxMultiSrc=" << m_maxMultiSrc);

        //form the multi src nodes based on the properties of the surface
        std::list<MultiSrcSurface*> multi_src_list = constructMultiSrcList(opaquereg_surf_list, m_maxMultiSrc);

        for (std::list<MultiSrcSurface*>::const_iterator multi_src_it = multi_src_list.begin();
                multi_src_it != multi_src_list.end(); multi_src_it++)
        {
            MultiSrcSurface* multi_src = *multi_src_it;

            if (1 == multi_src->m_surfaces.size())
            {
                std::list<Surface*>::const_iterator currentS = multi_src->m_surfaces.begin();

                if ((*currentS)->hasNativeContent() && (*currentS)->visibility && (*currentS)->opacity>0.0f)
                {
                    unsigned int layerId = (*currentS)->getContainingLayerId();
                    Layer* layer = m_baseWindowSystem->m_pScene->getLayer(layerId);

                    if (NULL == layer)
                    {
                        LOG_WARNING("Gal2dGraphicSystem",
                                "surface ID=" << (*currentS)->getID() << ", "
                                "is not attached to a valid layer");
                        continue;
                    }

                    if ((layer->getLayerType() != Software_2D) &&
                        (layer->getLayerType() != Software_2_5D))
                    {
                        LOG_WARNING("Gal2dGraphicSystem",
                                "layer ID=" << layer->getID() << ", "
                                "layer=" << layer << " is a non-SW layer");
                        continue;
                    }

                    beginLayer(layer);

                    LOG_INFO("Gal2dGraphicSystem",
                             "surface " << (*currentS)->getID() <<
                             " uses renderSurface");

                    renderSurface(*currentS);

                    endLayer();
                }
            }
            else
            {
                multiSrcBlit(multi_src->m_surfaces);
            }

            delete multi_src;
        }

        multi_src_list.clear();
    }
    else
    {
        for (SurfaceListConstIterator opaquereg_surf_it = opaquereg_surf_list.begin();
                opaquereg_surf_it != opaquereg_surf_list.end(); opaquereg_surf_it++)
        {
            if (clear && optimizeClear)
            {
                // Re-use the computeRegions code to fill out background areas not covered by
                // the first layer of surfaces (in case its not a single fullscreen surface).
                // Then perform an unblended draw for the whole first layer to "skip clear".

                //opaque region implementation does not take care of the computeRegions usecase

                std::list<MultiSurfaceRegion*> regions = computeRegions(opaquereg_surf_list, true); //do clear
                for (std::list<MultiSurfaceRegion*>::const_iterator region = regions.begin(); region != regions.end(); region++)
                {
                    renderRegion(*region, false); //don't blend
                }
            }
            else
            {
                unsigned int layerid = (*opaquereg_surf_it)->getContainingLayerId();
                Layer* layer = m_baseWindowSystem->m_pScene->getLayer(layerid);

                if (NULL == layer)
                {
                    LOG_WARNING("Gal2dGraphicSystem",
                            "surface ID=" << (*opaquereg_surf_it)->getID() << ", "
                            "is not attached to a valid layer");
                    continue;
                }
                if ((layer->getLayerType() != Software_2D) &&
                        (layer->getLayerType() != Software_2_5D))
                {
                    LOG_WARNING("Gal2dGraphicSystem",
                            "layer ID=" << layer->getID() << ", "
                            "layer=" << layer << " is a non-SW layer");
                    continue;
                }

                beginLayer(layer);
                renderSurface(*opaquereg_surf_it);
                endLayer();
            }
            clear = false;
        }
    }
}

void Gal2dGraphicSystem::startRepaintLoop(uint screenID)
{
    WaylandBaseWindowSystem* windowSystem;
    struct DrmOutput* output = NULL;
    struct timespec tVbl, tNow;
    struct timespec vbl2Now;
    gctUINT32 currentIndex;
    drmVBlank vbl;
    int ret;

    wl_list_for_each(output, &m_outputList, link) {
        if (screenID == output->screenID)
            break;
    }

    if (NULL == output)	{
        LOG_WARNING("Gal2dGraphicSystem","no drmOutput found for screenID "<< screenID);
        return;
    }

    /*repaint will be scheduled from page_flip_handler*/
    if (output->pageFlipPending) {
        return;
    }

    if (NULL == output->windowSystem) {
        LOG_WARNING("Gal2dGraphicSystem","no windowSystem present for output "<< output);
        return;
    }

    if (NULL == output->currentMode) {
        LOG_ERROR("Gal2dGraphicSystem","no currentMode for output "<< output);
        return;
    }

    /*to handle DPMS OFF case*/
    if (output->dpms_current_state != DRM_MODE_DPMS_ON)
        goto err_repaint_loop;

    vbl.request.type = DRM_VBLANK_RELATIVE;
    vbl.request.sequence = 0;
    vbl.request.signal = 0;

    ret = drmWaitVBlank(m_fdDev, &vbl);

    if ((ret == 0) && (vbl.reply.tval_sec > 0 || vbl.reply.tval_usec > 0)) {
        tVbl.tv_sec = vbl.reply.tval_sec;
        tVbl.tv_nsec = vbl.reply.tval_usec * 1000;

        clock_gettime(CLOCK_MONOTONIC, &tNow);
        timespecSub(&vbl2Now, &tNow, &tVbl);

        LOG_DEBUG("Gal2dGraphicSystem","VBlank Time: sec="<< tVbl.tv_sec << ", " <<
                    "nsec=" << tVbl.tv_nsec);

        LOG_DEBUG("Gal2dGraphicSystem","Current Time: sec="<< tNow.tv_sec << ", " <<
                    "nsec=" << tNow.tv_nsec);

        LOG_DEBUG("Gal2dGraphicSystem","Time Offset: sec="<< vbl2Now.tv_sec << ", " <<
                    "nsec=" << vbl2Now.tv_nsec);

        /*if we still have the time offset since the last vblank*/
        if (timespecToNsec(&vbl2Now) < output->currentMode->refresh_ns) {
            if (output->windowSystem) {
                LOG_DEBUG("Gal2dGraphicSystem","proceed with time_offset="<< timespecToNsec(&vbl2Now));
                windowSystem = static_cast<WaylandBaseWindowSystem*>(output->windowSystem);
                windowSystem->updateRepaintTimer(screenID, &tVbl, output->repaintWindowPeriodMS,
                        output->currentMode->refresh_ns, REPAINT_TIMER_UPDATE_INVALID);
                return;
            }
        }
        else
            LOG_DEBUG("Gal2dGraphicSystem","time offset not met OffsetNS="<< timespecToNsec(&vbl2Now));
    }

    if(output->currentIndex == 0)
    {
        currentIndex = output->dstNumSurfaces - 1;
    }
    else
    {
        currentIndex = output->currentIndex - 1;
    }

    /*do pageflip with the current frame-buffer
     *to get the exact vblank offset*/
    if (drmModePageFlip(m_fdDev,
                        output->crtcID,
                        output->buffer[currentIndex].fb,
                        DRM_MODE_PAGE_FLIP_EVENT,
                        output) < 0) {
        LOG_WARNING("Gal2dGraphicSystem","drmModePageFlip failed for output "<< output);
        goto err_repaint_loop;
    }

    return;

err_repaint_loop:
    windowSystem = static_cast<WaylandBaseWindowSystem*>(output->windowSystem);
    clock_gettime(CLOCK_MONOTONIC, &tNow);
    LOG_DEBUG("Gal2dGraphicSystem","updateRepaintTimer called with currentTime sec="<< tNow.tv_sec << ", " <<
              "nsec=" << tNow.tv_nsec);
    windowSystem->updateRepaintTimer(screenID, &tNow, output->repaintWindowPeriodMS,
            output->currentMode->refresh_ns, REPAINT_TIMER_UPDATE_INVALID);
}

bool Gal2dGraphicSystem::isFlipPending(uint screenID)
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, screen ID=" << screenID);
    bool ret = false;
    struct DrmOutput* output = NULL;
    wl_list_for_each(output, &m_outputList, link)
    {
        if (output->screenID != screenID)
        {
            continue;
        }
        if (output->pageFlipPending)
        {
            struct timespec cur_time;
            long elapced = 0;

            clock_gettime(CLOCK_MONOTONIC, &cur_time);

            elapced = ((cur_time.tv_sec - output->time_pageflip.tv_sec)*1000) +
                       ((cur_time.tv_nsec - output->time_pageflip.tv_nsec)/1000000);

            LOG_DEBUG("Gal2dGraphicSystem","pageflip pending for: "<<elapced<<" ms");

            if (elapced < MAX_TIME_TO_NEXT_PAGE_FLIP)
            {
                ret =  true;
            }
            else
            {
                char message[120];

                output->pageFlipPending = false;

                sprintf(message,"LAYERMANAGER: pageflippending reset manually after %d ms for screenid %d",
                                  MAX_TIME_TO_NEXT_PAGE_FLIP,output->screenID);
                LOG_ERROR("Gal2dGraphicSystem",""<< message);
                /*print entry in error memory*/
                log_to_errmem(message);
            }
        }
    }
    return ret;
}

bool Gal2dGraphicSystem::enableAlphaBlend(gcoSURF surface, float opacity, bool useSurfaceAlpha)
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, gcoSURF=" << surface << ", opacity=" << opacity << ", "
              "useSurfaceAlpha=" << useSurfaceAlpha);

    gceSTATUS status = gcvSTATUS_OK;
    if (gcvTRUE != m_hw2DPE20)
    {
        status = gcoSURF_EnableAlphaBlend(surface, (gctUINT8)(opacity * 255.0f), (gctUINT8)(opacity * 255.0f),
                                          gcvSURF_PIXEL_ALPHA_STRAIGHT, gcvSURF_PIXEL_ALPHA_STRAIGHT,
                                          gcvSURF_GLOBAL_ALPHA_OFF, gcvSURF_GLOBAL_ALPHA_OFF,
                                          gcvSURF_BLEND_ONE, gcvSURF_BLEND_INVERSED,
                                          gcvSURF_COLOR_MULTIPLY, gcvSURF_COLOR_STRAIGHT);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "gcoSURF_EnableAlphaBlend=" << status<< ", "
                        "gcoSURF=" << surface);
            return false;
        }

    }
    else
    {
        status = gco2D_EnableAlphaBlendAdvanced(m_gco2dEngine, gcvSURF_PIXEL_ALPHA_STRAIGHT, gcvSURF_PIXEL_ALPHA_STRAIGHT,
                                                gcvSURF_GLOBAL_ALPHA_SCALE, gcvSURF_GLOBAL_ALPHA_SCALE,
                                                gcvSURF_BLEND_STRAIGHT, gcvSURF_BLEND_INVERSED);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "gco2D_EnableAlphaBlendAdvanced=" << status << ", "
                        "gcoSURF=" << surface);
            return false;
        }

        status = gco2D_SetSourceGlobalColorAdvanced(m_gco2dEngine, ((gctUINT32)((gctUINT8)(opacity * 255.0f))<<24) | 0x00FFFFFF);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "gco2D_SetSourceGlobalColorAdvanced=" << status << ", "
                        "gcoSURF=" << surface);
            return false;
        }

        status = gco2D_SetTargetGlobalColorAdvanced(m_gco2dEngine,
                ((gctUINT32)((gctUINT8)( m_currentOutput->screen_alpha * 255.0f))<<24) | 0x00FFFFFF);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "gco2D_SetTargetGlobalColorAdvanced=" << status);
        }

        if (true == useSurfaceAlpha)
        {
            status = gco2D_SetPixelMultiplyModeAdvanced(m_gco2dEngine, gcv2D_COLOR_MULTIPLY_ENABLE, gcv2D_COLOR_MULTIPLY_DISABLE,
                                                    gcv2D_GLOBAL_COLOR_MULTIPLY_ALPHA, gcv2D_COLOR_MULTIPLY_DISABLE);
        }
        else
        {
            status = gco2D_SetPixelMultiplyModeAdvanced(m_gco2dEngine, gcv2D_COLOR_MULTIPLY_DISABLE, gcv2D_COLOR_MULTIPLY_DISABLE,
                                                    gcv2D_GLOBAL_COLOR_MULTIPLY_ALPHA, gcv2D_COLOR_MULTIPLY_DISABLE);
        }
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "gco2D_SetPixelMultiplyModeAdvanced=" << status << ", "
                        "gcoSURF=" << surface);
            return false;
        }
    }

    return true;
}

class SurfaceLocker
{
public:
    SurfaceLocker(gcoSURF surf)
        : _surf(surf)
        , _status(gcvSTATUS_OK)
        , _locked(false)
    {
        _phy[0] = _phy[1] = _phy[2]  = 0;
        _log[0] = _log[1]= _log[2]= NULL;
        _status = gcoSURF_Lock(_surf, _phy, _log);
        _locked = (_status == gcvSTATUS_OK);
    }

    ~SurfaceLocker() { unlock(); }

    gctUINT32 *phy() { return _phy; }
    gctPOINTER *log() { return _log; }
    gceSTATUS status() { return _status; }
    bool locked() { return _locked; }

    bool unlock() {
        if (locked()) {
            _status = gcoSURF_Unlock(_surf, _log);
            _locked = false;

            return _status == gcvSTATUS_OK;
        }

        return true;
    }

private:
    gcoSURF _surf;
    gceSTATUS _status;
    bool _locked;
    gctUINT32 _phy[3];
    gctPOINTER _log[3];
};

bool Gal2dGraphicSystem::isYUVFomat(gceSURF_FORMAT format)
{
	bool isYUVFomat = false;

	// When in YUV, we have to use filter blitting anyways.
	switch(format) {
	case gcvSURF_YUY2:
	case gcvSURF_UYVY:
	case gcvSURF_YV12:
	case gcvSURF_I420:
	case gcvSURF_NV12:
	case gcvSURF_NV21:
	case gcvSURF_NV16:
	case gcvSURF_NV61:
	case gcvSURF_YVYU:
	case gcvSURF_VYUY:
		isYUVFomat = true;
	    break;
	default:
		isYUVFomat = false;
	    break;
	}

	return isYUVFomat;
}

bool Gal2dGraphicSystem::useFilterBlit(gceSURF_FORMAT format)
{
    bool useFilterBlit = false;

    // When in YUV, we have to use filter blitting anyways.
    switch(format) {
    case gcvSURF_YUY2:
    case gcvSURF_UYVY:
    case gcvSURF_YV12:
    case gcvSURF_I420:
    case gcvSURF_NV12:
    case gcvSURF_NV21:
    case gcvSURF_NV16:
    case gcvSURF_NV61:
    case gcvSURF_YVYU:
    case gcvSURF_VYUY:
        useFilterBlit = true;
        break;
    default:
        // When RGB, we need to use filter blit when asked to do nice scaling
        useFilterBlit = mConfiguration->getFilterKernelSize() > 0;
        break;
    }

    return useFilterBlit;
}

void Gal2dGraphicSystem::multiSrcBlit(SurfaceList surfaces)
{
    LOG_DEBUG("Gal2dGraphicSystem",
                "Called, no. layers=" << surfaces.size());

    int number_of_surfaces, i = 0;
    unsigned int surface_mask = 0;
    gceSTATUS status = gcvSTATUS_OK;
    gcsRECT clipRect = {0, 0, 0, 0};

    number_of_surfaces = surfaces.size();

    if (!number_of_surfaces)
    {
        LOG_ERROR("Gal2dGraphicSystem",
                    "MultiSrcBlit is called with " << number_of_surfaces << "surfaces");
        return;
	}

    for(std::list<Surface*>::const_iterator currentS = surfaces.begin(); currentS != surfaces.end(); currentS++)
    {
        LOG_INFO("Gal2dGraphicSystem",
                    "surface " << (*currentS)->getID() <<
                    " uses multiSrcBlit");

        unsigned int layerId = (*currentS)->getContainingLayerId();
        Layer* layer = m_baseWindowSystem->m_pScene->getLayer(layerId);

        if (NULL == layer)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                    "surface ID=" << (*currentS)->getID() << ", "
                    "is not attached to a valid layer");
            continue;
        }

        if ((layer->getLayerType() != Software_2D) &&
            (layer->getLayerType() != Software_2_5D))
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "layer ID=" << layer->getID() << ", "
                        "layer=" << layer << " is a non-SW layer");
            continue;
        }

        // check if surface is cropped completely, if so then skip rendering
        if ((*currentS)->isCropped())
            continue; // skip rendering of this surface, because it is cropped by layer source region

        gco2D_SetCurrentSourceIndex(m_gco2dEngine, i);

        /* Bind surface and set section */
        if (false == m_binder->bindSurfaceTexture((*currentS)))
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "LM surface not successfully bound, "
                        "surface ID=" << (*currentS)->getID());
            continue;
        }

        FloatRectangle srcRegion = (*currentS)->getCorrectedSrcRegion();
        FloatRectangle dstRegion = (*currentS)->getTargetDestinationRegion();
        gcsRECT srcClipRect = {0, 0, 0, 0};
        srcClipRect.left = (gctINT32)dstRegion.x;
        srcClipRect.top = (gctINT32)dstRegion.y;
        srcClipRect.right = (gctINT32)(dstRegion.width + dstRegion.x);
        srcClipRect.bottom = (gctINT32)(dstRegion.height + dstRegion.y);

        /*trim destination rectangle to screen resolution*/
        if ( srcClipRect.left > m_displayWidth)
        {
            srcClipRect.left = m_displayWidth -1;
        }
        if ( srcClipRect.top > m_displayHeight)
        {
            srcClipRect.top = m_displayHeight -1;
        }
        if ( srcClipRect.right > m_displayWidth)
        {
            srcClipRect.right = m_displayWidth;
        }
        if ( srcClipRect.bottom > m_displayHeight)
        {
            srcClipRect.bottom = m_displayHeight;
        }

        GcoSurfWaylandPlatformSurface* nativeSurface = (GcoSurfWaylandPlatformSurface*)(*currentS)->platform;
        if (!nativeSurface || !nativeSurface->m_surface)
        {
            LOG_WARNING("Gal2dGraphicSystem", "bad nativeSurface");
            continue;
        }

        status = gco2D_SetClipping(m_gco2dEngine, &srcClipRect);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to set clipping, "
                    "dstRect: (left=" << srcClipRect.left << ", "
                    "right=" << srcClipRect.right << ", "
                    "bottom=" << srcClipRect.bottom << ", "
                    "top=" << srcClipRect.top << "), "
                    "target CRTC ID=" << m_currentOutput->crtcID << ", "
                    "connector ID=" << m_currentOutput->connectorID << ", "
                    "screen ID=" << m_currentOutput->screenID << ", "
                    "surface ID=" << (*currentS)->getID() << ", "
                    "containing layer ID="
                    << (*currentS)->getContainingLayerId() << ", "
                    "gcoSURF="
                    << m_currentOutput->currentTargetSurface);
            return;
        }

        // enable alpha only if the source surface supports it.
        gcsSURF_FORMAT_INFO_PTR formatInfo = 0;
        bool useSurfaceAlpha = false;

        status = gcoSURF_GetFormatInfo(nativeSurface->m_surface, &formatInfo);
        if (gcvSTATUS_OK == status && formatInfo != 0 && formatInfo->fmtClass == gcvFORMAT_CLASS_RGBA)
        {
            useSurfaceAlpha = (formatInfo->u.rgba.alpha.width > 0) &&
                                  ((formatInfo->u.rgba.alpha.width & gcvCOMPONENT_DONTCARE) == 0);
        }

        LOG_INFO("Gal2dGraphicSystem",
                "surface opacity=" << (*currentS)->getOpacity() <<
                ",layer opacity=" << layer->getOpacity() <<
                ",useSurfaceAlpha=" << useSurfaceAlpha <<
                ",m_hw2DPE20=" << m_hw2DPE20 <<
                ",m_currentOutput->screen_alpha=" << m_currentOutput->screen_alpha <<
                ",multisrc-index=" << i);

        if ((false == enableAlphaBlend(
                m_currentOutput->currentTargetSurface,
                (*currentS)->getOpacity() * layer->getOpacity(), useSurfaceAlpha)) ||
                (gcvSTATUS_OK != gco2D_SetROP(m_gco2dEngine, 0xCC, 0xCC)))
        {
            LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to enable alpha blend, "
                    "dstRect: (left=" << srcClipRect.left << ", "
                    "right=" << srcClipRect.right << ", "
                    "bottom=" << srcClipRect.bottom << ", "
                    "top=" << srcClipRect.top << "), "
                    "target CRTC ID=" << m_currentOutput->crtcID << ", "
                    "connector ID=" << m_currentOutput->connectorID << ", "
                    "screen ID=" << m_currentOutput->screenID << ", "
                    "surface ID=" << (*currentS)->getID() << ", "
                    "containing layer ID="
                    << (*currentS)->getContainingLayerId() << ", "
                    "gcoSURF="
                    << m_currentOutput->currentTargetSurface);
        }

        gceSURF_FORMAT format;
        status = gcoSURF_GetFormat(nativeSurface->m_surface, gcvNULL, &format);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem", "Failed to get format");
            continue;
        }

        // for YUV formats bindSurfaceTexture is not setting the source
        // so we have to set the source info here.
        if(isYUVFomat(format))
        {
            gctUINT32 strides_num = 1, address_num = 1;
            gctUINT width, height;
            gctINT strides[3];
            gceTILING tiling;

            gcoSURF_GetTiling(nativeSurface->m_surface, &tiling);

            status = gcoSURF_GetAlignedSize(nativeSurface->m_surface,
                                            &width, &height, strides);

            SurfaceLocker srcLock(nativeSurface->m_surface);
            if (!srcLock.locked())
            {
                LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to lock source, "
                        "surface ID=" << (*currentS)->getID() << ", "
                        "native surface ID=" << nativeSurface->surfaceId << ", "
                        "containing layer ID="
                        << (*currentS)->getContainingLayerId() << ", "
                        "wayland platform surface="
                        << nativeSurface << ", "
                        "status=" << srcLock.status());
                continue;
            }

            /* GetAlignedSize sets only strides[0]; compute remaining ones if necessary.
             * While at it, compute num planes and swap pointers when necessary. */
            switch (format) {
                case gcvSURF_NV12:
                case gcvSURF_NV16:
                    strides[1] = strides[0];
                    strides_num = address_num = 2;
                    break;

                case gcvSURF_YV12:
                    strides[2] = strides[1] = strides[0] / 2;
                    strides_num = address_num = 3;
                    break;

                default:
                    strides_num = address_num = 1;
                    break;
            }

            status = gco2D_SetGenericSource(m_gco2dEngine, srcLock.phy(), address_num,
                                (gctUINT32_PTR)strides, strides_num, tiling, format,
                                gcvSURF_0_DEGREE, width, height);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to set source, "
                        "surface ID=" << (*currentS)->getID() << ", "
                        "native surface ID=" << nativeSurface->surfaceId << ", "
                        "containing layer ID="
                        << (*currentS)->getContainingLayerId() << ", "
                        "wayland platform surface="
                        << nativeSurface << ", "
                        "status=" << srcLock.status());
            }

            gcsRECT srcRect = {0, 0, 0, 0};
            srcRect.left   = (gctINT32)srcRegion.x;
            srcRect.top    = (gctINT32)srcRegion.y;
            srcRect.right  = (gctINT32)(srcRegion.width + srcRegion.x);
            srcRect.bottom = (gctINT32)(srcRegion.height + srcRegion.y);

            status = gco2D_SetSource(m_gco2dEngine, &srcRect);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "gco2D_SetSource failed, " <<
                            "left=" << srcRect.left << ", " <<
                            "top=" << srcRect.top << ", " <<
                            "right=" << srcRect.right << ", " <<
                            "bottom=" << srcRect.bottom);
            }
        }

        m_binder->unbindSurfaceTexture((*currentS));

        surface_mask |= (1 << i);
        i++;
    }

    LOG_DEBUG("Gal2dGraphicSystem",
              "surface_mask=" << surface_mask);

    gctUINT dst_width = 0;
    gctUINT dst_height = 0;
    gctINT dst_stride = 0;

    status = gcoSURF_GetAlignedSize(m_currentOutput->currentTargetSurface,
                                    &dst_width, &dst_height, &dst_stride);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                "Failed to get aligned dst size, "
                "status=" << status);
        return;
    }

    clipRect.left = 0;
    clipRect.top = 0;
    clipRect.right = dst_width;
    clipRect.bottom = dst_height;

    gco2D_SetClipping(m_gco2dEngine, &clipRect);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                "gco2D_SetClipping Failed, "
                "status=" << status);
        return;
    }

    //do multi-src blit
    status = gco2D_MultiSourceBlit(m_gco2dEngine, surface_mask, &clipRect, 1);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                "gco2D_MultiSourceBlit Failed, "
                "status=" << status);
    }

    i = 0;
    for(std::list<Surface*>::const_iterator currentS = surfaces.begin(); currentS != surfaces.end(); currentS++)
    {
        gco2D_SetCurrentSourceIndex(m_gco2dEngine, i);

        status = gco2D_DisableAlphaBlend(m_gco2dEngine);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                    "Failed disable alpha blend, "
                    "surface ID=" << (*currentS)->getID() << ", "
                    "containing layer ID="
                    << (*currentS)->getContainingLayerId() << ", "
                    "status=" << status);
        }
    }

    gco2D_SetCurrentSourceIndex(m_gco2dEngine, 0);
}

void Gal2dGraphicSystem::renderSurface(Surface* surface)
{
    LOG_DEBUG("Gal2dGraphicSystem", "Called, surface ID=" << surface->getID());

    gceSTATUS status = gcvSTATUS_OK;
    gctUINT width;
    gctUINT height;
    gctINT strides[3];

   // check if surface is cropped completely, if so then skip rendering
    if (surface->isCropped())
        return; // skip rendering of this surface, because it is cropped by layer source region
    /* Bind surface and set section */
    if (false == m_binder->bindSurfaceTexture(surface))
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "LM surface not successfully bound, "
                    "surface ID=" << surface->getID());
        return;
    }

    FloatRectangle srcRegion = surface->getCorrectedSrcRegion();
    FloatRectangle dstRegion = surface->getTargetDestinationRegion();
    gcsRECT dstRect = {0, 0, 0, 0};
    gcsRECT clipRect = {0, 0, 0, 0};
    dstRect.left   = clipRect.left =(gctINT32)dstRegion.x;
    dstRect.top    = clipRect.top = (gctINT32)dstRegion.y;
    dstRect.right  = clipRect.right = (gctINT32)(dstRegion.width + dstRegion.x);
    dstRect.bottom = clipRect.bottom = (gctINT32)(dstRegion.height + dstRegion.y);

    /*trim destination rectangle to screen resolution*/
    if ( clipRect.left > m_displayWidth)
    {
        clipRect.left = m_displayWidth -1;
    }
    if ( clipRect.top > m_displayHeight)
    {
        clipRect.top = m_displayHeight -1;
    }
    if ( clipRect.right > m_displayWidth)
    {
        clipRect.right = m_displayWidth;
    }
    if ( clipRect.bottom > m_displayHeight)
    {
        clipRect.bottom = m_displayHeight;
    }
    status = gco2D_SetClipping(m_gco2dEngine, &clipRect);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to set clipping, "
                    "dstRect: (left=" << dstRect.left << ", "
                    "right=" << dstRect.right << ", "
                    "bottom=" << dstRect.bottom << ", "
                    "top=" << dstRect.top << "), "
                    "target CRTC ID=" << m_currentOutput->crtcID << ", "
                    "connector ID=" << m_currentOutput->connectorID << ", "
                    "screen ID=" << m_currentOutput->screenID << ", "
                    "surface ID=" << surface->getID() << ", "
                    "containing layer ID="
                    << surface->getContainingLayerId() << ", "
                    "gcoSURF="
                    << m_currentOutput->currentTargetSurface);
        return;
    }

    GcoSurfWaylandPlatformSurface* nativeSurface = (GcoSurfWaylandPlatformSurface*)surface->platform;
    if (!nativeSurface || !nativeSurface->m_surface)
    {
        LOG_WARNING("Gal2dGraphicSystem", "bad nativeSurface");
        return;
    }

    // enable alpha only if the source surface supports it.
    gcsSURF_FORMAT_INFO_PTR formatInfo = 0;
    bool useSurfaceAlpha = false;

    status = gcoSURF_GetFormatInfo(nativeSurface->m_surface, &formatInfo);
    if (gcvSTATUS_OK == status && formatInfo != 0 && formatInfo->fmtClass == gcvFORMAT_CLASS_RGBA)
    {
        useSurfaceAlpha = (formatInfo->u.rgba.alpha.width > 0) &&
                          ((formatInfo->u.rgba.alpha.width & gcvCOMPONENT_DONTCARE) == 0);
    }

    if (false == enableAlphaBlend(
            m_currentOutput->currentTargetSurface,
			(surface)->getOpacity() * m_currentLayer->getOpacity(), useSurfaceAlpha))
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to enable alpha blend, "
                    "dstRect: (left=" << dstRect.left << ", "
                    "right=" << dstRect.right << ", "
                    "bottom=" << dstRect.bottom << ", "
                    "top=" << dstRect.top << "), "
                    "target CRTC ID=" << m_currentOutput->crtcID << ", "
                    "connector ID=" << m_currentOutput->connectorID << ", "
                    "screen ID=" << m_currentOutput->screenID << ", "
                    "surface ID=" << surface->getID() << ", "
                    "containing layer ID="
                    << surface->getContainingLayerId() << ", "
                    "gcoSURF="
                    << m_currentOutput->currentTargetSurface);
    }

    gceSURF_FORMAT format;

    status = gcoSURF_GetFormat(nativeSurface->m_surface, gcvNULL, &format);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem", "Failed to get format");
        return;
    }

    LOG_DEBUG("Gal2dGraphicSystem",
              "Rendering - dstRect: (left=" << dstRect.left << ", "
              "right=" << dstRect.right << ", "
              "bottom=" << dstRect.bottom << ", "
              "top=" << dstRect.top << "), "
              "target CRTC ID=" << m_currentOutput->crtcID << ", "
              "connector ID=" << m_currentOutput->connectorID << ", "
              "screen ID=" << m_currentOutput->screenID << ", "
              "surface ID=" << surface->getID() << ", "
              "native surface ID=" << nativeSurface->surfaceId << ", "
              "containing layer ID=" << surface->getContainingLayerId() << ", "
              "gcoSURF=" << m_currentOutput->currentTargetSurface << ", "
              "wayland platform surface=" << nativeSurface << ", "
              "frame counter=" << surface->frameCounter << ", "
              "native format code=" << format);

	status = gcoSURF_GetAlignedSize(nativeSurface->m_surface,
                                        &width, &height, strides);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get aligned size, "
                    "native surface ID=" << nativeSurface->surfaceId << ", "
                    "containing layer ID="
                    << nativeSurface->s->getContainingLayerId() << ", "
                    "wayland platform surface="
                    << nativeSurface << ", "
                    "status=" << status);
        return;
    }

    // One path for YUV surfaces, one path for RGB surfaces.

    if (useFilterBlit(format))
    {
        // YUV path.
        gctUINT32 planes_num;

        SurfaceLocker srcLock(nativeSurface->m_surface);
        if (!srcLock.locked())
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to set stretch factor, "
                        "surface ID=" << surface->getID() << ", "
                        "native surface ID=" << nativeSurface->surfaceId << ", "
                        "containing layer ID="
                        << surface->getContainingLayerId() << ", "
                        "wayland platform surface="
                        << nativeSurface << ", "
                        "status=" << srcLock.status());
            return;
        }

        /* GetAlignedSize sets only strides[0]; compute remaining ones if necessary.
         * While at it, compute num planes and swap pointers when necessary. */
    switch (format) {
    case gcvSURF_NV12:
    case gcvSURF_NV16:
        strides[1] = strides[0];
        planes_num = 2;
        break;

    case gcvSURF_YV12:
        strides[2] = strides[1] = strides[0] / 2;
        planes_num = 3;
        break;

    default:
        planes_num = 1;
        break;
    }

        gceSURF_FORMAT dst_format;

        status = gcoSURF_GetFormat(m_currentOutput->currentTargetSurface, gcvNULL, &dst_format);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to get dst format, "
                        "surface ID=" << surface->getID() << ", "
                        "native surface ID=" << nativeSurface->surfaceId << ", "
                        "containing layer ID="
                        << surface->getContainingLayerId() << ", "
                        "wayland platform surface="
                        << nativeSurface << ", "
                        "status=" << status);
            return;
        }

        gctUINT dst_width = 0;
        gctUINT dst_height = 0;
        gctINT dst_stride = 0;

        status = gcoSURF_GetAlignedSize(m_currentOutput->currentTargetSurface,
                                        &dst_width, &dst_height, &dst_stride);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to get aligned dst size, "
                        "surface ID=" << surface->getID() << ", "
                        "containing layer ID="
                        << surface->getContainingLayerId() << ", "
                        "wayland platform surface="
                        << nativeSurface << ", "
                        "status=" << status);
            return;
        }

        SurfaceLocker dstLock(m_currentOutput->currentTargetSurface);
        if (not dstLock.locked())
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to lock dst surface, "
                        "surface ID=" << surface->getID() << ", "
                        "containing layer ID="
                        << surface->getContainingLayerId() << ", "
                        "wayland platform surface="
                        << nativeSurface << ", "
                        "status=" << dstLock.status());
            return;
        }

        gcsRECT srcRect = {0, 0, 0, 0};
        srcRect.left   = (gctINT32)srcRegion.x;
        srcRect.top    = (gctINT32)srcRegion.y;
        srcRect.right  = (gctINT32)(srcRegion.width + srcRegion.x);
        srcRect.bottom = (gctINT32)(srcRegion.height + srcRegion.y);

        gctUINT32 unsigned_strides[3];
        unsigned_strides[0] = (gctUINT32) strides[0];
        unsigned_strides[1] = (gctUINT32) strides[1];
        unsigned_strides[2] = (gctUINT32) strides[2];

        // Modify clip rect for FilterBlitEx2. We're using dstSubRect to clip
        // whose coordinates are relative to dstRect
        clipRect.left = 0;
        clipRect.top = 0;
        clipRect.right = min (m_displayWidth - dstRect.left, dstRect.right - dstRect.left);
        clipRect.bottom = min (m_displayHeight - dstRect.top, dstRect.bottom - dstRect.top);

        status = gco2D_FilterBlitEx2(
                     m_gco2dEngine,
                     srcLock.phy(),
                     planes_num,
                     unsigned_strides,
                     planes_num,
                     gcvLINEAR,                // TODO: src tiling
                     format,
                     gcvSURF_0_DEGREE,
                     width,
                     height,
                     &srcRect,
                     dstLock.phy(),
                     1,                        // DestAddressNum
                     (gctUINT32_PTR)&dst_stride,
                     1,
                     gcvLINEAR,                // TODO: dest tiling
                     dst_format,
                     gcvSURF_0_DEGREE,
                     dst_width,
                     dst_height,
                     &dstRect,
                     &clipRect
                     );
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "FilterBlitEx2 failed, "
                        "surface ID=" << surface->getID() << ", "
                        "native surface ID=" << nativeSurface->surfaceId << ", "
                        "containing layer ID="
                        << surface->getContainingLayerId() << ", "
                        "wayland platform surface="
                        << nativeSurface << ", "
                        "status=" << status);
            return;
        }

        if (not dstLock.unlock())
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to unlock dst surface, "
                        "surface ID=" << surface->getID() << ", "
                        "native surface ID=" << nativeSurface->surfaceId << ", "
                        "containing layer ID="
                        << surface->getContainingLayerId() << ", "
                        "wayland platform surface="
                        << nativeSurface << ", "
                        "status=" << dstLock.status());
            return;
        }

        if (not srcLock.unlock())
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to unlock surface, "
                        "surface ID=" << surface->getID() << ", "
                        "native surface ID=" << nativeSurface->surfaceId << ", "
                        "containing layer ID="
                        << surface->getContainingLayerId() << ", "
                        "wayland platform surface="
                        << nativeSurface << ", "
                        "status=" << srcLock.status());
            return;
        }
    } else {
        // RGB path.
        if (((srcRegion.x == dstRegion.x) && (srcRegion.y == dstRegion.y)
         && (srcRegion.width == dstRegion.width) && (srcRegion.height == dstRegion.height))
         || ((1 > srcRegion.width) || (2 > dstRegion.width) || (1 > dstRegion.height) || (2 > dstRegion.height)))
        {
            status = gco2D_Blit(m_gco2dEngine, 1, &dstRect, 0xCC, 0xAA, m_currentOutput->surfaceFormat);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "Failed to Blit, "
                            "surface ID=" << surface->getID() << ", "
                            "native surface ID=" << nativeSurface->surfaceId << ", "
                            "containing layer ID="
                            << surface->getContainingLayerId() << ", "
                            "wayland platform surface="
                            << nativeSurface << ", "
                            "status=" << status);
                return;
            }
        }
        else
        {
            gctUINT horFactor = (((gctUINT)srcRegion.width - 1) << 16) / ((gctUINT)dstRegion.width - 1);
            gctUINT verFactor = (((gctUINT)srcRegion.height - 1) << 16) / ((gctUINT)dstRegion.height - 1);
            status = gco2D_SetStretchFactors(m_gco2dEngine, horFactor, verFactor);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "Failed to set stretch factor, "
                            "surface ID=" << surface->getID() << ", "
                            "native surface ID="
                            << nativeSurface->surfaceId << ", "
                            "containing layer ID="
                            << surface->getContainingLayerId() << ", "
                            "wayland platform surface="
                            << nativeSurface << ", "
                            "horFactor=" << horFactor << ", "
                            "verFactor=" << verFactor << ", "
                            "status=" << status);
                return;
            }
            status = gco2D_StretchBlit(m_gco2dEngine, 1, &dstRect, 0xCC, 0xAA, m_currentOutput->surfaceFormat);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "Failed to StretchBlit, "
                            "surface ID=" << surface->getID() << ", "
                            "native surface ID="
                            << nativeSurface->surfaceId << ", "
                            "containing layer ID="
                            << surface->getContainingLayerId() << ", "
                            "wayland platform surface="
                            << nativeSurface << ", "
                            "status=" << status);
                return;
            }
        }
    }

    status = gco2D_DisableAlphaBlend(m_gco2dEngine);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed disable alpha blend, "
                    "surface ID=" << surface->getID() << ", "
                    "native surface ID=" << nativeSurface->surfaceId << ", "
                    "containing layer ID="
                    << surface->getContainingLayerId() << ", "
                    "wayland platform surface="
                    << nativeSurface << ", "
                    "status=" << status);
    }
#if 0
    if (gcvTRUE == m_hw2DPE20)
    {
        status = gco2D_SetPixelMultiplyModeAdvanced(m_gco2dEngine, gcv2D_COLOR_MULTIPLY_DISABLE, gcv2D_COLOR_MULTIPLY_DISABLE,
                                                gcv2D_GLOBAL_COLOR_MULTIPLY_DISABLE, gcv2D_COLOR_MULTIPLY_DISABLE);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "gco2D_SetPixelMultiplyModeAdvanced:" << status);
        }
    }
#endif

    surface->frameCounter++;
    surface->fpsFrameCounter++;

    m_binder->unbindSurfaceTexture(surface);
}

// For a given area of the screen, render the appropriate part of all the
// surfaces listed in the provided SurfaceList.
//     returns true if rendering succeeds
//     returns false if rendering was aborted.  e.g. no shader was available
bool Gal2dGraphicSystem::renderSurfaces(SurfaceList surfaces, FloatRectangle targetDestination, bool blend)
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, No. surfaces=" << surfaces.size() << ", "
              "targetDestination.x=" << targetDestination.x << ", "
              "targetDestination.y=" << targetDestination.y << ", "
              "targetDestination.width=" << targetDestination.width << ", "
              "targetDestination.height=" << targetDestination.height << ", "
              "blend=" << blend);

    gceSTATUS status = gcvSTATUS_OK;
    blend = blend;

    if (surfaces.size() > MAX_MULTI_SURFACE)
    {
        return false;
    }

    // update all common uniforms, scale values to display size
    // offsets are generated w.r.t lower left corner (following GL conventions)
    gcsRECT clipRect = {0, 0, 0, 0};
    clipRect.left   = (gctINT32)targetDestination.x;
    clipRect.top    = (gctINT32)targetDestination.y;
    clipRect.right  = (gctINT32)(targetDestination.width - targetDestination.x);
    clipRect.bottom = (gctINT32)(targetDestination.height - targetDestination.y);
    /*trim destination rectangle to screen resolution*/
    if ( clipRect.left > m_displayWidth)
    {
        clipRect.left = m_displayWidth -1;
    }
    if ( clipRect.top > m_displayHeight)
    {
        clipRect.top = m_displayHeight -1;
    }
    if ( clipRect.right > m_displayWidth)
    {
        clipRect.right = m_displayWidth;
    }
    if ( clipRect.bottom > m_displayHeight)
    {
        clipRect.bottom = m_displayHeight;
    }
    status = gco2D_SetClipping(m_gco2dEngine, &clipRect);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to set clipping"
                    " clipping area: "
                    "left=" << clipRect.left << ", "
                    "right=" << clipRect.right << ", "
                    "bottom=" << clipRect.bottom << ", "
                    "top=" << clipRect.top);
        return false;
    }

    // update per-texture uniforms
    int surfacenum = 0;
    for (SurfaceListConstIterator surface = surfaces.begin(); surface != surfaces.end(); surface++, surfacenum++)
    {
        LOG_DEBUG("Gal2dGraphicSystem",
                  "Rendering, surface ID=" << (*surface)->getID() << ", "
                  "containing layer ID="
                  << (*surface)->getContainingLayerId());

        /* Bind surface and set section */
        if (false == m_binder->bindSurfaceTexture(*surface))
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Surface not successfully bound, surface ID="
                        << (*surface)->getID() << ", "
                        "containing layer ID="
                        << (*surface)->getContainingLayerId());
            return false;
        }

        // setting opacity
        int layerId = (*surface)->getContainingLayerId();
        Layer* layer = m_baseWindowSystem->m_pScene->getLayer(layerId);

        if (false == enableAlphaBlend(
                m_currentOutput->currentTargetSurface,
				(*surface)->getOpacity() * layer->getOpacity(), true))
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to enable alpha blend"
                        ", surface ID=" << (*surface)->getID()
                        << ", containing layer ID="
                        << (*surface)->getContainingLayerId());
        }

        FloatRectangle srcRegion = (*surface)->getTargetSourceRegion();
        FloatRectangle dstRegion = (*surface)->getTargetDestinationRegion();
        gcsRECT dstRect = {0, 0, 0, 0};
        dstRect.left   = (gctINT32)dstRegion.x;
        dstRect.top    = (gctINT32)dstRegion.y;
        dstRect.right  = (gctINT32)(dstRegion.width - dstRegion.x);
        dstRect.bottom = (gctINT32)(dstRegion.height - dstRegion.y);

        GcoSurfWaylandPlatformSurface* nativeSurface = (GcoSurfWaylandPlatformSurface*)(*surface)->platform;
        if (!nativeSurface || !nativeSurface->m_surface)
        {
            LOG_WARNING("Gal2dGraphicSystem", "bad nativeSurface");
            return false;
        }

        gctUINT width;
        gctUINT height;
        gctINT strides[3];

        status = gcoSURF_GetAlignedSize(nativeSurface->m_surface,
                                        &width, &height, strides);

        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to get aligned dst size, "
                        "surface ID=" << (*surface)->getID() << ", "
                        "containing layer ID="
                        << (*surface)->getContainingLayerId() << ", "
                        "wayland platform surface="
                        << nativeSurface << ", "
                        "status=" << status);
            return false;
        }

        if (((srcRegion.x == dstRegion.x) && (srcRegion.y == dstRegion.y)
         && (srcRegion.width == dstRegion.width) && (srcRegion.height == dstRegion.height))
         || ((1 > srcRegion.width) || (2 > dstRegion.width) || (1 > dstRegion.height) || (2 > dstRegion.height)))
        {
            status = gco2D_Blit(m_gco2dEngine, 1, &dstRect, 0xCC, 0xAA, m_currentOutput->surfaceFormat);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "Failed to Blit, "
                            "surface ID=" << (*surface)->getID() << ", "
                            "containing layer ID="
                            << (*surface)->getContainingLayerId() << ", "
                            "status=" << status << ", "
                            "Destination Area: "
                            "left=" << dstRect.left << ", "
                            "right=" << dstRect.right << ", "
                            "bottom=" << dstRect.bottom << ", "
                            "top=" << dstRect.top);
                return false;
            }
        }
        else
        {
            gctUINT horFactor = (((gctUINT)srcRegion.width - 1) << 16) / ((gctUINT)dstRegion.width - 1);
            gctUINT verFactor = (((gctUINT)srcRegion.height - 1) << 16) / ((gctUINT)dstRegion.height - 1);
            status = gco2D_SetStretchFactors(m_gco2dEngine, horFactor, verFactor);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "Failed to set stretch factor, "
                            "surface ID=" << (*surface)->getID() << ", "
                            "containing layer ID="
                            << (*surface)->getContainingLayerId() << ", "
                            "status=" << status << ", "
                            "horizontal factor=" << horFactor << ", "
                            "vertical factor=" << verFactor);
                return false;
            }
            status = gco2D_StretchBlit(m_gco2dEngine, 1, &dstRect, 0xCC, 0xAA, m_currentOutput->surfaceFormat);
            if (gcvSTATUS_OK != status)
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "Failed to StretchBlit, "
                            "surface ID=" << (*surface)->getID() << ", "
                            "containing layer ID="
                            << (*surface)->getContainingLayerId() << ", "
                            "status=" << status);
                return false;
            }
        }

        (*surface)->frameCounter++;
        (*surface)->fpsFrameCounter++;

        m_binder->unbindSurfaceTexture(*surface);
    }

    return true;
}

void Gal2dGraphicSystem::renderRegion(MultiSurfaceRegion* region, bool blend)
{
    // Render all surfaces in this region, performing as many rendering
    // passes as necessary, each with up to MAX_MULTI_SURFACE surfaces.
    SurfaceList surfaceBatch;

    LOG_DEBUG("Gal2dGraphicSystem", "Called, "
              "start coordinate: "
              "(x=" << region->m_rect.x << ", y=" << region->m_rect.y << "), "
              "width=" << region->m_rect.width << ", "
              "height=" << region->m_rect.height);

    for(SurfaceListConstIterator currentS = region->m_surfaces.begin(); currentS != region->m_surfaces.end(); currentS++)
    {
        surfaceBatch.push_back(*currentS);

        // Render when batch size reaches maximum, then start a new batch
        if (surfaceBatch.size() == MAX_MULTI_SURFACE)
        {
            if (!renderSurfaces(surfaceBatch, region->m_rect, blend))
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "Failed to render region "
                            "(multi-pass/multi-texture), "
                            "surface ID=" << (*currentS)->getID() << ", "
                            "containing layer ID="
                            << (*currentS)->getContainingLayerId());
                return;
            }
            surfaceBatch.clear();
            blend = true; // Do blend for subsequent batches
        }
    }

    // Finish remaining work, take care of clear-color regions now also
    if (!blend || surfaceBatch.size() > 0)
    {
        if (!renderSurfaces(surfaceBatch, region->m_rect, blend))
        {
            LOG_DEBUG("Gal2dGraphicSystem",
                      "Failed to render region (multi-pass/multi-texture) "
                      "Start Coordinate: "
                      "(x=" << region->m_rect.x << ", "
                      "y=" << region->m_rect.y << "), "
                      "width=" << region->m_rect.width << ", "
                      "height=" << region->m_rect.height
);
            return;
        }
    }
}

struct OrderedSurface {
    Surface *surface;
    int depth;
    OrderedSurface(Surface *s, int d):surface(s),depth(d) {}
    bool operator<(OrderedSurface rhs) const { return depth < rhs.depth; }
    bool operator==(OrderedSurface rhs) const { return surface == rhs.surface; }
};

struct RegionLimit {
    float value; // Location on either X or Y axis
    bool entry;  // 1=in, 0=out
    OrderedSurface orderedSurface;
    RegionLimit(float v, bool e, OrderedSurface s):value(v),entry(e),orderedSurface(s) {}
    bool operator<(RegionLimit rhs) const { return value < rhs.value; }
};

// This API will group the surfaces based on the feasibility for opaqueRegion.
// The opaqueRegion for the surface need to be set by the application.
// Currently, the opaqueRegion is supported for applications using libapx
std::list<Surface*> Gal2dGraphicSystem::getOpaqueRegSurfList(LayerList layers)
{
	LayerListConstIterator layer;
	SurfaceList opaquereg_surf_list;
	for(layer = layers.begin(); layer != layers.end(); layer++)
	{
		if ((*layer)->visibility == false || (*layer)->getOpacity() <= 0.0f)
		{
			continue;
		}

		if ((*layer)->getAllSurfaces().size() == 0) continue;

		SurfaceList surfaces = (*layer)->getAllSurfaces();
		SurfaceListConstIterator surface;
		for(surface = surfaces.begin(); surface != surfaces.end(); surface++)
		{
			if (!(*surface)->hasNativeContent())
			{
				continue;
			}

			if ((*surface)->visibility == false || (*surface)->getOpacity() <= 0.0f)
			{
				continue;
			}

			FloatRectangle SurfRegion = (*surface)->getTargetDestinationRegion();
			if(((*surface)->m_opaqueRegion) &&
					(SurfRegion.width == m_displayWidth && SurfRegion.height == m_displayHeight))
			{
				opaquereg_surf_list.clear();
				opaquereg_surf_list.push_back(*surface);

				LOG_INFO("Gal2dGraphicSystem",
				    "surface " << (*surface)->getID() <<
				    " is added to the opaque region surface list ");
			}
			else
			{
				opaquereg_surf_list.push_back(*surface);
			}
		}
	}

	return opaquereg_surf_list;
}

// This API will group the surfaces based on the feasibility for multi-src blit.
// If two or more consecutive surfaces are eligible to be used as multi-src then
// those surfaces will be formed as a single node. If any of the surface's properties
// are not eligible for multi-src blit that will be created as a seperate node and
// will be rendered by using renderSurface API.
std::list<MultiSrcSurface*> Gal2dGraphicSystem::constructMultiSrcList(std::list<Surface*> opaqueRegSurfList, int maxSurf)
{
    std::list<MultiSrcSurface*> multi_src_list;
    MultiSrcSurface *multi_src_node = NULL;

    SurfaceList surfaces = opaqueRegSurfList;
    SurfaceListConstIterator surface;
    for(surface = surfaces.begin(); surface != surfaces.end(); surface++)
    {
        if (!(*surface)->hasNativeContent())
        {
            continue;
        }

        if ((*surface)->visibility == false || (*surface)->getOpacity() <= 0.0f)
        {
            continue;
        }

        FloatRectangle srcRegion = (*surface)->getCorrectedSrcRegion();
        FloatRectangle dstRegion = (*surface)->getTargetDestinationRegion();
        gctINT32 srcX = (gctINT32)srcRegion.x;
        gctINT32 srcY = (gctINT32)srcRegion.y;
        gctINT32 dstX = (gctINT32)dstRegion.x;
        gctINT32 dstY = (gctINT32)dstRegion.y;
        gctINT32 srcWidth = (gctINT32)srcRegion.width;
        gctINT32 srcHeight = (gctINT32)srcRegion.height;
        gctINT32 dstWidth = (gctINT32)dstRegion.width;
        gctINT32 dstHeight = (gctINT32)dstRegion.height;
        bool suitable = false;
        bool is_multi_planar_yuv = false;

        //constraints of multi-src blit
        //max surfaces - 4/8
        //scaling not supported
        //source and dest rect should be same
        //only a single multi planar yuv format surface is allowed
        suitable = ((srcX == dstX) && (srcY == dstY) &&
		            (srcWidth == dstWidth) && (srcHeight == dstHeight));

        GcoSurfWaylandPlatformSurface* nativeSurface = (GcoSurfWaylandPlatformSurface*)(*surface)->platform;
        if (!nativeSurface || !nativeSurface->m_surface)
        {
            continue;
        }

        gceSURF_FORMAT format;
        gceSTATUS status = gcoSURF_GetFormat(nativeSurface->m_surface, gcvNULL, &format);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem", "Failed to get format");
            continue;
        }

        switch (format)
        {
            case gcvSURF_I420:
            case gcvSURF_YV12:
            case gcvSURF_NV16:
            case gcvSURF_NV12:
            case gcvSURF_NV61:
            case gcvSURF_NV21:
                is_multi_planar_yuv = true;
                break;

            default:
                is_multi_planar_yuv = false;
                break;
        }

        if (multi_src_node && multi_src_node->supported &&
            suitable && (multi_src_node->m_surfaces.size() < (unsigned int)maxSurf) &&
            !(multi_src_node->has_multi_planar_yuv && is_multi_planar_yuv))
        {
            multi_src_node->m_surfaces.push_back(*surface);
            multi_src_node->has_multi_planar_yuv = is_multi_planar_yuv;

            LOG_INFO("Gal2dGraphicSystem",
                     "surface " << (*surface)->getID() <<
                     " is added to the multi-src node " <<
                     multi_src_node <<
                     ", multi_planar_yuv_flag=" <<
                     multi_src_node->has_multi_planar_yuv);
        }
        else
        {
            multi_src_node = new MultiSrcSurface();
            multi_src_node->has_multi_planar_yuv = is_multi_planar_yuv;
            multi_src_node->m_surfaces.push_back(*surface);
            multi_src_node->supported = suitable;
            multi_src_list.push_back(multi_src_node);

            LOG_INFO("Gal2dGraphicSystem",
                     "surface " << (*surface)->getID() <<
                     " is created as new multi-src node " <<
                     multi_src_node << " supported_flag=" <<
                     multi_src_node->supported <<
                     ", multi_planar_yuv_flag=" <<
                     multi_src_node->has_multi_planar_yuv);
        }
    }

    return multi_src_list;
}

// Creates a list of non-overlapping screen-space regions, each with a list of
// surfaces that reside in that area of the screen.  If 'clear' is true, the
// list of screen regions are guaranteed to collectively cover the entire
// screen space (empty surface lists are used to indicate regions with only
// background color visible).  When 'clear' is false, the set of regions will
// only cover screen space occupied by these layers' surfaces.
std::list<MultiSurfaceRegion*> Gal2dGraphicSystem::computeRegions(std::list<Surface*> opaqueRegSurfList, bool clear)
{
    /*
     * Example: on a 10x6 screen, with two surfaces A,B
     *
     * Surface A = (1,0)-(9,3)
     * Surface B = (3,2)-(9,5)
     *
     * Sorted edge lists (note X=9 is listed twice):
     *     xlimits = 1 A(in), 3 B(in), 9 A(out), 9 B(out)
     *     ylimits = 0 A(in), 2 B(in), 3 A(out), 5 B(out)
     *
     *  X     1     3                 9
     * Y
     * 0     _ ________________________ _
     *     |  |                       |  |
     *     |  |           A           |  |
     *     |  |                       |  |
     * 2   | _|_ _ _ _________________|_ |
     *     |  |  A  |       A+B       |  |
     * 3   | _|_____|_________________|_ |
     *     |        |                 |  |
     *     |        |        B        |  |
     *     |        |                 |  |
     * 5   | _ _ _ _|_________________|_ |
     *     |                             |
     *     | _ _ _ _ _ _ _ _ _ _ _ _ _ _ |
     *
     * Without clear, the process will create 4 regions. 3 single surface regions
     * and one dual-surface region.
     *
     * With clear, the edges of the screen are also added into xlimits and ylimits,
     * so 11 regions are created. 3 single-surface, 1-dual-surface, and 7 no-surface.
     */

    LOG_DEBUG("Gal2dGraphicSystem", "Called");

    std::list<MultiSurfaceRegion*> regions;
    std::vector<RegionLimit> xlimits;
    std::vector<RegionLimit> ylimits;

    // If clear is needed enable a dummy surface so that additional zero-surface
    // regions will be created around the edges to fill screen (when necessary)
    if (clear)
    {
        xlimits.push_back(RegionLimit(0,               true,  OrderedSurface(NULL, 0)));
        xlimits.push_back(RegionLimit(m_displayWidth,  false, OrderedSurface(NULL, 0)));
        ylimits.push_back(RegionLimit(0,               true,  OrderedSurface(NULL, 0)));
        ylimits.push_back(RegionLimit(m_displayHeight, false, OrderedSurface(NULL, 0)));
    }

    // Add the edges of the visible surfaces into xlimits and ylimits.  Each edge is flagged
    // as either an "IN" (left or top edge), or an "OUT" (right or bottom edge).  Layer depth is
    // also stored in the edge information.
    int depth = 0;

    SurfaceList surfaces = opaqueRegSurfList;
    SurfaceListConstIterator surface;
    for(surface = surfaces.begin(); surface != surfaces.end(); surface++)
    {
        if (!(*surface)->hasNativeContent())
        {
            continue;
        }

        if ((*surface)->visibility == false || (*surface)->getOpacity() <= 0.0f)
        {
            continue;
        }

        depth++;

        FloatRectangle rect = (*surface)->getTargetDestinationRegion();

        xlimits.push_back(RegionLimit(rect.x,               true,  OrderedSurface((*surface), depth)));
        xlimits.push_back(RegionLimit(rect.x + rect.width,  false, OrderedSurface((*surface), depth)));
        ylimits.push_back(RegionLimit(rect.y,               true,  OrderedSurface((*surface), depth)));
        ylimits.push_back(RegionLimit(rect.y + rect.height, false, OrderedSurface((*surface), depth)));
    }

    // Early exit if there were no visible surfaces
    if (xlimits.size() < 2 || ylimits.size() < 2)
    {
        return regions; //empty
    }

    // Sort the surface edges prior to traversal
    std::sort(xlimits.begin(), xlimits.end());
    std::sort(ylimits.begin(), ylimits.end());

    /* Create the regions by stepping through rows and columns, keeping
     * track of which surfaces are present using IN/OUT edge information.
     *
     * Foreach ylimit {
     *     If the ylimit defines a new row {
     *         Foreach xlimit {
     *             If the xlimit defines a new column {
     *                 If xlimit is for a surface not in "rowsurfaces", ignore it
     *                 Create new region containing "regionsurfaces"
     *                 Add the surface to the current "regionsurfaces" list if "IN"
     *                 Remove from the current "regionsurfaces" list if "OUT"
     *             }
     *         }
     *     }
     *
     *     Add the surface to the current "rowsurfaces" list if "IN"
     *     Remove from the current "rowsurfaces" list if "OUT"
     * }
     *
     * Care must be taken with duplicate xlimit or ylimit edges. Like in the
     * example, both A and B are OUT at X=9, so there are two xlimit values
     * with X=9.
     */

    // The set of surfaces which are present somewhere in the current row
    std::set<Surface*> rowsurfaces;

    int x1 = -1; // Left of current region
    int x2 = -1; // Right of current region
    int y1 = -1; // Top of current row/region
    int y2 = -1; // Bottom of current row/region

    // The first row on the screen starts at the very top
    y1 = ylimits[0].value;
    for (unsigned y = 0; y < ylimits.size(); y++)
    {
        // The current row is between y1 and y2
        y2 = ylimits[y].value;

        // When y1 and y2 are the same, this isn't a new row yet.  Do update
        // "rowsurfaces", but don't yet step across the row.
        if (y2 != y1)
        {
            // Process the new row.  The list of regionsurfaces starts empty
            std::list<OrderedSurface> regionsurfaces;

            // 1st region in the row starts at the far left
            x1 = xlimits[0].value;
            for (unsigned x = 0; x < xlimits.size(); x++)
            {
                // Skip x edges that don't pertain to surfaces within the
                // current row.  This coallesces rows into as few regions as
                // possible. Columns are not coallesced though.
                if (rowsurfaces.find(xlimits[x].orderedSurface.surface) == rowsurfaces.end())
                {
                    continue;
                }
                // If this x edge defines the right side of a new region,
                // create that region.
                x2 = xlimits[x].value;
                if (x2 != x1)
                {
                    // A new region is ready to be created.  It exists between
                    // (x1,y1)-(x2,y2) and has the surfaces which currently are
                    // in "regionsurfaces".
                    MultiSurfaceRegion *region = new MultiSurfaceRegion();
                    region->m_rect = FloatRectangle(x1, y1, x2-x1, y2-y1);
                    std::list<OrderedSurface>::iterator si;
                    for (si = regionsurfaces.begin(); si != regionsurfaces.end(); si++)
                    {
                        if ((*si).surface)
                        {
                            region->m_surfaces.push_back((*si).surface);
                        }
                    }
                    // Add the new region to the list we return.  Only create
                    // empty regions when 'clear' was requested
                    if (clear || regionsurfaces.size() > 0)
                    {
                        regions.push_back(region);
                    }
                    // The right side of this region becomes the left side of
                    // the next region
                    x1 = x2;
                }

                // Update the list of surfaces in the current region
                if (xlimits[x].entry)
                {
                    // This xlimit is the left side of a surface, add it to
                    // the list.  Keep the surfaces in this list sorted by
                    // depth so when the region object is created, it will
                    // be able to render with the surfaces layered properly.
                    bool inserted = false;
                    std::list<OrderedSurface>::iterator si;
                    for (si = regionsurfaces.begin(); si != regionsurfaces.end(); si++)
                    {
                        if ((*si).depth >= xlimits[x].orderedSurface.depth)
                        {
                            regionsurfaces.insert(si, xlimits[x].orderedSurface);
                            inserted = true;
                            break;
                        }
                    }
                    if (!inserted)
                    {
                        regionsurfaces.push_back(xlimits[x].orderedSurface);
                    }
                }
                else
                {
                    // This xlimit is the right side of a surface, remove it
                    // from the list
                    regionsurfaces.remove(xlimits[x].orderedSurface);
                }
            }
            // The bottom of this row becomes the top of the next row
            y1 = y2;
        }

        // Update the list of surfaces which are present in this current row
        if (ylimits[y].entry)
        {
            rowsurfaces.insert(ylimits[y].orderedSurface.surface);
        }
        else
        {
            rowsurfaces.erase(ylimits[y].orderedSurface.surface);
        }
    }

    return regions;
}

bool Gal2dGraphicSystem::initGal2d(int displayWidth, int displayHeight)
{
    gceSTATUS status = gcvSTATUS_OK;
    m_displayWidth = displayWidth;
    m_displayHeight = displayHeight;

    LOG_DEBUG("Gal2dGraphicSystem", "Called");
    do
    {
        status = gcoOS_Construct(gcvNULL, &m_gcoOs);
        if ((gcvSTATUS_OK != status) || (NULL == m_gcoOs))
        {
            LOG_WARNING("Gal2dGraphicSystem", "Failed to create gcoOS object");
            return false;
        }

        status = gcoHAL_Construct(gcvNULL, m_gcoOs, &m_gcoHal);
        if ((gcvSTATUS_OK != status) || (NULL == m_gcoHal))
        {
            LOG_WARNING("Gal2dGraphicSystem", "Failed to create gcoHAL object");
            return false;
        }

        /* Check the HW feature. */
        m_hw2DPE20 = gcoHAL_IsFeatureAvailable(m_gcoHal, gcvFEATURE_2DPE20);
        LOG_INFO("Gal2dGraphicSystem",
                    "H/W Feature(2DPE20)=" << m_hw2DPE20);

        status = gcoHAL_Get2DEngine(m_gcoHal, &m_gco2dEngine);
        if ((gcvSTATUS_OK != status) || (NULL == m_gco2dEngine))
        {
            LOG_WARNING("Gal2dGraphicSystem", "Failed to get 2DEngine object");
            return false;
        }

        status = gcoHAL_SetHardwareType(m_gcoHal, gcvHARDWARE_2D);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem", "Failed to set Hardware type");
            return false;
        }

        // Enable XRGB formats, XRGB is handled as full ARGB "for historical reasons"
        // by the engine.
        status = gco2D_SetStateU32(m_gco2dEngine, gcv2D_STATE_XRGB_ENABLE, gcvTRUE);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem", "Failed to enable XRGB support");

            // Not returning false here, our alpha channel work-around will still
            // make XRGB work correctly.
        }

        if (mConfiguration->getFilterKernelSize() > 0)
        {
            status = gco2D_SetKernelSize(m_gco2dEngine,
                                         mConfiguration->getFilterKernelSize(),
                                         mConfiguration->getFilterKernelSize());
            if ((gcvSTATUS_OK != status))
            {
                LOG_WARNING("Gal2dGraphicSystem", "Failed to set 2D engine kernel size");
                mConfiguration->disableFilterBlit();
            }
        }

        if (mConfiguration->getEnableMultiSrcBlit())
        {
            if ((gcoHAL_IsFeatureAvailable(m_gcoHal, gcvFEATURE_2D_MULTI_SOURCE_BLT_EX) == gcvTRUE) ||
                (gcoHAL_IsFeatureAvailable(m_gcoHal, gcvFEATURE_2D_MULTI_SRC_BLT_TO_UNIFIED_DST_RECT) == gcvTRUE))
            {
                m_maxMultiSrc = 8;

                LOG_INFO("Gal2dGraphicSystem", "MultiSrcBlit Supported, m_maxMultiSrc=" << m_maxMultiSrc);
            }
            else if (gcoHAL_IsFeatureAvailable(m_gcoHal, gcvFEATURE_2D_MULTI_SOURCE_BLT) == gcvTRUE)
            {
                m_maxMultiSrc = 4;

                LOG_INFO("Gal2dGraphicSystem", "MultiSrcBlit Supported, m_maxMultiSrc=" << m_maxMultiSrc);
            }
            else
            {
                m_maxMultiSrc = 0;
                mConfiguration->setEnableMultiSrcBlit(0);

                LOG_INFO("Gal2dGraphicSystem", "multi source blit is not supported");
            }
        }
        else
        {
            m_maxMultiSrc = 0;
        }

        return true;

    } while(0);

    releaseResources();
    return false;
}

void Gal2dGraphicSystem::resize(int displayWidth, int displayHeight)
{
    LOG_DEBUG("Gal2dGraphicSystem", "Called");

    m_displayWidth = displayWidth;
    m_displayHeight = displayHeight;
}

void Gal2dGraphicSystem::startScreenshotThread(FileThreadObject *screenshot)
{
    int ret = 0;
    LOG_DEBUG("Gal2dGraphicSystem", "Called");
    if (m_currentScreenshotThread != NULL)
    {
        pthread_join(m_fileThread, NULL);
        delete m_currentScreenshotThread;
    }
    m_currentScreenshotThread = screenshot;
    ret = pthread_create(&m_fileThread, NULL, &FileThreadObject::run,
                         (void*)screenshot);
    if (ret != 0)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Error creating file writing thread!");
        sem_post(&m_workerSem);
    }
}

void Gal2dGraphicSystem::writeLayerFile(const char *filename,
                                        int layerWidth, int layerHeight,
                                        int blankWidth, int blankHeight,
                                        int bottomSize, int bufferDepth,
                                        size_t start_offset,
                                        size_t display_row_width,
                                        size_t layer_row_width,
                                        char* buffer)
{
    (void)layerWidth;

    std::ofstream outfile(filename, std::ofstream::binary);
    if (!outfile.good())
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to open file, path=" << filename);
    }
    else
    {
        for (int i=0; i < layerHeight; i++)
        {
            size_t offset = start_offset + display_row_width * i;
            char *ptr = buffer + offset;
            // Write the buffer for that line
            outfile.write(ptr, layer_row_width);
            // Write zeroes for the part of the line that's off-screen
            if (blankWidth != 0)
            {
                for (int i = 0; i < blankWidth * bufferDepth; i++)
                {
                    const char zero = 0;
                    outfile.write(&zero, 1);
                }
            }
            if (outfile.fail())
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "Error occurred while writing output file, "
                            "path= " << filename);
                break;
            }
        }
        if (blankHeight != 0)
        {
            for (int i = 0; i < bottomSize; i++)
            {
                const char zero = 0;
                outfile.write(&zero, 1);
            }
        }
        outfile.close();
    }
}

void Gal2dGraphicSystem::writeFile(const char* filename, int size, char* buffer, gcoSURF *tempSurf )
{
        std::ofstream outfile(filename, std::ofstream::binary);
        if (!outfile.good())
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to open file, path=" << filename);
        }
        else
        {
            outfile.write(buffer, size);
            if (outfile.fail())
            {
                LOG_WARNING("Gal2dGraphicSystem",
                            "Error occurred while writing output file, path="
                            << filename);
            }
            outfile.close();
        }

        if (tempSurf != NULL)
        {
            gcoSURF_Destroy(tempSurf[0]);
            delete[] tempSurf;
        }
}

void Gal2dGraphicSystem::writeBMPFile(const char* filename, int width, int height, gceSURF_FORMAT format, char* buffer)
{
    char *dst_buf = (char*)malloc(width*height*3);
    if (!dst_buf)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to allocate memory");
        return;
    }

    if (format == gcvSURF_R5G6B5)
    {
        RGB565toBGR888(dst_buf, buffer, width, height);
    }
    else if (format == gcvSURF_X8R8G8B8)
    {
        RGBA8888toBGR888(dst_buf, buffer, width, height);
    }
    else
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Unsupported format of display buffer, "
                    "format=" << format);
    }
    writeBitmap(filename, dst_buf, width, height);
    free(dst_buf);
}

void Gal2dGraphicSystem::saveScreenShotOfFramebuffer(const std::string& fileToSave,
                                                     DumpOutputType outputType)
{
    LOG_DEBUG("Gal2dGraphicSystem",
              "Called, fileToSave=" << fileToSave << ", "
              "outputType=" << outputType);

    char *filename = new char[fileToSave.length()+1];
    size_t buffer_size;

    gceSTATUS status = gcvSTATUS_OK;
    gctINT stride;
    gctUINT width;
    gctUINT height;
    gceSURF_FORMAT format;
    gctUINT32 phyAddr;
    gctPOINTER logAddr;

    FileThreadObject *thread = NULL;

    strcpy(filename, fileToSave.c_str());

    int realIndex = m_currentOutput->currentIndex;

    if (!m_currentOutput->isScreenshot)
    {
        // Got here without switchScreen, check it's ok to take a screenshot
        if (sem_trywait(&m_workerSem) < 0)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Currently taking screenshot!");
            delete[] filename;
            return;
        }
        if (!m_currentOutput->pageFlipPending)
        {
            // Page flip handler would have incremented the currentIndex, we
            // want the current state so deal with that here
            realIndex = realIndex - 1;
            if (realIndex < 0)
                realIndex = m_currentOutput->dstNumSurfaces - 1;
            setTargetSurface(m_currentOutput->dstSurfaces[realIndex]);
        }
    }

    if (outputType == DUMP_UNKNOWN)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Called with unknown dump output type");
        goto err_surf;
    }

    status = gcoSURF_GetAlignedSize(m_currentOutput->dstSurfaces[realIndex], &width, &height, &stride);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get aligned size, "
                    "status=" << status);
        goto err_surf;
    }

    status = gcoSURF_GetFormat(m_currentOutput->dstSurfaces[realIndex], gcvNULL, &format);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get format, "
                    "status=" << status);
        goto err_surf;
    }

    status = gcoSURF_Lock(m_currentOutput->dstSurfaces[realIndex],
                          &phyAddr,
                          (gctPOINTER*)&logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to lock surface, "
                    "status=" << status);
        goto err_surf;
    }

    status = gcoSURF_Unlock(m_currentOutput->dstSurfaces[realIndex],
                           (gctPOINTER)logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to unlock surface");
        goto err_surf;
    }

    if (!m_currentOutput->isScreenshot)
    {
        // We got here without switching the screen, so this means we are
        // making a full screen dump, change the renderoutput to a screenshot
        // output and copy the display buffer onto it
        gcsRECT srcRect;
        DrmOutput *renderOutput = m_currentOutput;

        renderOutput = findScreenshotOutput(m_currentOutput->screenID);
        if (renderOutput == NULL)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to find screenshot output");
            goto err_surf;
        }

        status = gcoSURF_Lock(renderOutput->dstSurfaces[renderOutput->currentIndex],
                              &phyAddr,
                              (gctPOINTER*)&logAddr);

        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to lock render surface");
            goto err_surf;
        }

        srcRect.left   = (gctINT32)0;
        srcRect.top    = (gctINT32)0;
        srcRect.right  = (gctINT32)width;
        srcRect.bottom = (gctINT32)height;
        status = gcoSURF_FilterBlit(m_currentOutput->dstSurfaces[realIndex],
                                    renderOutput->dstSurfaces[renderOutput->currentIndex],
                                    &srcRect,
                                    &srcRect,
                                    NULL);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to blit the surface onto the screenshot surface");
            goto err_surf;
        }
        setTargetSurface(renderOutput->dstSurfaces[renderOutput->currentIndex]);
        status = gcoSURF_Unlock(renderOutput->dstSurfaces[renderOutput->currentIndex], (gctPOINTER)logAddr);
        if (gcvSTATUS_OK != status)
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to unlock surface");
            goto err_surf;
        }
    }

    flushAndCommit();

    if (format != gcvSURF_R5G6B5 && format != gcvSURF_X8R8G8B8)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Unexpected format, value=" << format << ", "
                    "instead of gcvSURF_R5G6B5 or gcvSURF_X8R8G8B8");
        goto err_surf;
    }

    if (outputType == DUMP_RAW)
    {
        buffer_size = stride * height;
        thread = new FileThreadObject(filename, buffer_size, (char*)logAddr, this, &m_workerSem);
    }
    else if (outputType == DUMP_BMP)
    {
        thread = new FileThreadObject(filename, width, height, (char*)logAddr, format, this, &m_workerSem);
    }
    else
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Unexpected output, type=" << outputType);

        goto err_surf;
    }

    startScreenshotThread(thread);

    return;

err_surf:
    delete[] filename;
    sem_post(&m_workerSem);
    return;
}

void Gal2dGraphicSystem::saveScreenShotOfLayer(const std::string& fileToSave,
                                               Layer* layer)
{
    gceSTATUS status = gcvSTATUS_OK;
    gctINT stride;
    gctUINT width;
    gctUINT height;
    gceSURF_FORMAT format;
    gctUINT32 phyAddr;
    gctPOINTER logAddr;

    FileThreadObject *thread = NULL;

    size_t start_offset;
    size_t display_row_width;
    size_t layer_row_width;

    Vector2 pos;
    Vector2 dim;
    unsigned int layerX;
    unsigned int layerY;
    unsigned int layerWidth;
    unsigned int layerHeight;
    unsigned int blankWidth;
    unsigned int blankHeight;
    unsigned int bottomSize;
    unsigned int bufferDepth;
    char *filename = new char[fileToSave.length()+1];

    strcpy(filename, fileToSave.c_str());
    flushAndCommit();

    // Get layer dimensions
    if (layer == NULL)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Layer is NULL, file path=" << fileToSave);
        goto err_surface;
    }
    else
    {
        LOG_DEBUG("Gal2dGraphicSystem",
                  "Called, fileToSave=" << fileToSave << ", "
                  "layer ID=" << layer->getID());
    }

    pos = layer->getPosition();
    dim = layer->getDimension();
    layerX = pos.val1;
    layerY = pos.val2;
    layerWidth = dim.val1;
    layerHeight = dim.val2;
    blankWidth = 0;
    blankHeight = 0;

    // Get the buffer and its dimensions

    status = gcoSURF_GetAlignedSize(m_currentOutput->dstSurfaces[m_currentOutput->currentIndex],
                                    &width, &height, &stride);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get aligned size, layer ID=" << layer->getID());
        goto err_surface;
    }
    status = gcoSURF_GetFormat(m_currentOutput->dstSurfaces[m_currentOutput->currentIndex],
                               gcvNULL, &format);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get format, "
                    "layer ID=" << layer->getID());
        goto err_surface;
    }
    if (format == gcvSURF_R5G6B5)
    {
        bufferDepth = 2;
    }
        else if (format == gcvSURF_X8R8G8B8)
    {
        bufferDepth = 4;
    }
    else
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Format is of unexpected type, value= " << format << ", "
                    "layer ID=" << layer->getID());
        goto err_surface;
    }

    status = gcoSURF_Lock(m_currentOutput->dstSurfaces[m_currentOutput->currentIndex],
                          &phyAddr, (gctPOINTER*)&logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to lock surface, "
                    "layer ID=" << layer->getID());
        goto err_surface;
    }

    // Avoid dumping coordinates outside the screen
    if (layerX + layerWidth > width)
    {
        blankWidth = layerX + layerWidth - width;
        layerWidth = width - layerX;
    }
    if (layerY + layerHeight > height)
    {
        blankHeight = layerY + layerHeight - height;
        layerHeight = height - layerY;
    }

    start_offset = (width * layerY + layerX) * bufferDepth;
    bottomSize = blankHeight * (layerWidth + blankWidth) * bufferDepth;
    display_row_width = width * bufferDepth;
    layer_row_width = layerWidth * bufferDepth;

    LOG_DEBUG("Gal2dGraphicSystem",
              "dumping subdivision with "
              "pos x=" << layerX << ", pos y=" << layerY << ", "
              "width=" << layerWidth << ", "
              "height=" << layerHeight << ", "
              "stride=" << stride << ", "
              "screen width=" << width << ", "
              "screen height=" << height << ", "
              "blank width=" << blankWidth << ", "
              "blank height=" << blankHeight << ", "
              "depth=" << bufferDepth);

    status = gcoSURF_Unlock(m_currentOutput->dstSurfaces[m_currentOutput->currentIndex],
                            (gctPOINTER)logAddr);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to unlock surface, "
                    "layer ID=" << layer->getID());
        goto err_surface;
    }

    thread = new FileThreadObject(filename,
                                  layerWidth,
                                  layerHeight,
                                  blankWidth,
                                  blankHeight,
                                  bottomSize,
                                  bufferDepth,
                                  start_offset,
                                  display_row_width,
                                  layer_row_width,
                                  (char*)logAddr,
                                  this,
                                  &m_workerSem);

    startScreenshotThread(thread);

    return;

err_surface:
    delete[] filename;
    sem_post(&m_workerSem);
    return;
}

static uint32_t getDepthBitsFromFormat(gceSURF_FORMAT format)
{
    LOG_DEBUG("Gal2dGraphicSystem", "Called, format=" << format);

    switch(format)
    {
        case gcvSURF_UNKNOWN:
            LOG_WARNING("Gal2dGraphicSystem", "Format is unknown");
            return 0;
        case gcvSURF_A2R2G2B2:
        case gcvSURF_R3G3B2:
            return 8;
        case gcvSURF_A8R3G3B2:
        case gcvSURF_X4R4G4B4:
        case gcvSURF_A4R4G4B4:
        case gcvSURF_R4G4B4A4:
        case gcvSURF_X1R5G5B5:
        case gcvSURF_A1R5G5B5:
        case gcvSURF_R5G5B5A1:
        case gcvSURF_R5G6B5:
        case gcvSURF_R5G5B5X1:
        case gcvSURF_R4G4B4X4:
        case gcvSURF_A4B4G4R4:
        case gcvSURF_A1B5G5R5:
        case gcvSURF_B5G6R5:
        case gcvSURF_B4G4R4A4:
        case gcvSURF_B5G5R5A1:
        case gcvSURF_X4B4G4R4:
        case gcvSURF_X1B5G5R5:
        case gcvSURF_B4G4R4X4:
        case gcvSURF_B5G5R5X1:
            return 16;
        case gcvSURF_R8G8B8:
        case gcvSURF_B8G8R8:
            return 24;
        case gcvSURF_X8R8G8B8:
        case gcvSURF_A8R8G8B8:
        case gcvSURF_R8G8B8A8:
        case gcvSURF_G8R8G8B8:
        case gcvSURF_R8G8B8G8:
        case gcvSURF_X2R10G10B10:
        case gcvSURF_A2R10G10B10:
        case gcvSURF_R8G8B8X8:
        case gcvSURF_X8B8G8R8:
        case gcvSURF_A8B8G8R8:
        case gcvSURF_A2B10G10R10:
        case gcvSURF_B8G8R8X8:
        case gcvSURF_B8G8R8A8:
        case gcvSURF_X2B10G10R10:
            return 32;
        case gcvSURF_X12R12G12B12:
        case gcvSURF_A12R12G12B12:
        case gcvSURF_B16G16R16:
            return 48;
        case gcvSURF_X16R16G16B16:
        case gcvSURF_A16R16G16B16:
        case gcvSURF_X16B16G16R16:
        case gcvSURF_A16B16G16R16:
            return 64;
        case gcvSURF_B32G32R32:
            return 96;
        case gcvSURF_A32R32G32B32:
        case gcvSURF_X32B32G32R32:
        case gcvSURF_A32B32G32R32:
            return 128;

        default:
            LOG_WARNING("Gal2dGraphicSystem",
                        "Format (code=" << format << ") "
                        "is un-handled or unexpected");
            return 0;
    }
}

void Gal2dGraphicSystem::saveScreenShotOfSurface(const std::string& fileToSave,
                                                 Surface* surface)
{
    FloatRectangle surfaceDim;
    unsigned int layerId;
    Layer *layer;
    Rectangle layerDim;

    uint32_t surfaceScreenX;
    uint32_t surfaceScreenY;
    uint32_t surfaceWidth;
    uint32_t surfaceHeight;
    uint32_t blankWidth;
    uint32_t blankHeight;
    uint32_t bufferDepth;

    size_t start_offset;
    size_t bottomSize;
    size_t display_row_width;
    size_t surface_row_width;

    gceSTATUS status = gcvSTATUS_OK;
    gcoSURF dispSurf;
    gctINT stride;
    gctUINT width;
    gctUINT height;
    gceSURF_FORMAT format;
    gctUINT32 phyAddr;
    gctPOINTER logAddr;

    char *filename = new char[fileToSave.length()+1];
    FileThreadObject *thread = NULL;

    strcpy(filename, fileToSave.c_str());

    if (surface == NULL)
    {
        LOG_WARNING("Gal2dGraphicSystem", "surface is NULL");
        goto err_surf;
    }
    else
    {
        LOG_DEBUG("Gal2dGraphicSystem",
                  "Called, file path= " << fileToSave << ", "
                  "surface ID=" << surface->getID());
    }

    // Get surface dimensions
    surfaceDim = surface->getDestinationRegion();

    layerId = surface->getContainingLayerId();
    layer = m_baseWindowSystem->m_pScene->getLayer(layerId);
    layerDim = layer->getDestinationRegion();

    // Get surface's screen position
    surfaceScreenX = surfaceDim.x + layerDim.x;
    surfaceScreenY = surfaceDim.y + layerDim.y;

    surfaceWidth = surfaceDim.width;
    surfaceHeight = surfaceDim.height;
    blankWidth = 0;
    blankHeight = 0;

    // Clip against layer
    if (surfaceDim.x + surfaceWidth > layerDim.width)
    {
        blankWidth = surfaceDim.x + surfaceDim.width - layerDim.width;
        surfaceWidth = layerDim.width - surfaceDim.x;
    }
    if (surfaceDim.y + surfaceHeight > layerDim.height)
    {
        blankHeight = surfaceDim.y + surfaceDim.height - layerDim.height;
        surfaceHeight = layerDim.height - surfaceDim.y;
    }

    // Clip against screen
    if (surfaceScreenX + surfaceWidth > (unsigned int) m_displayWidth)
    {
        blankWidth = surfaceScreenX + surfaceDim.width - m_displayWidth;
        surfaceWidth = m_displayWidth - surfaceScreenX;
    }
    if (surfaceScreenY + surfaceHeight > (unsigned int) m_displayHeight)
    {
        blankHeight = surfaceScreenY + surfaceDim.height - m_displayHeight;
        surfaceHeight = m_displayHeight - surfaceScreenY;
    }

    // Get display surface info
    dispSurf =
        m_currentOutput->dstSurfaces[m_currentOutput->currentIndex];

    LOG_DEBUG("Gal2dGraphicSystem",
              "gcoSURF=" << dispSurf << ", "
              "surface ID=" << surface->getID());

    status = gcoSURF_GetAlignedSize(dispSurf, &width, &height, &stride);
    if (status != gcvSTATUS_OK)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get aligned size for gcoSURF=" << dispSurf);
        goto err_surf;
    }
    status = gcoSURF_GetFormat(dispSurf, gcvNULL, &format);
    if (status != gcvSTATUS_OK)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get format for gcoSURF=" << dispSurf);
        goto err_surf;
    }

    bufferDepth = getDepthBitsFromFormat(format) / 8;
    if (bufferDepth == 0)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get depth from format=" << format);
        goto err_surf;
    }

    LOG_DEBUG("Gal2dGraphicSystem",
              "Dumping surface with dimensions(x=" << surfaceDim.x << ", "
              "y=" << surfaceDim.y << ", "
              "w=" << surfaceDim.width << ", "
              "h=" << surfaceDim.height << ") "
              "inside layer, ID=" << layerId << ", "
              "which has dimensions (x=" << layerDim.x << ", "
              "y=" << layerDim.y << ", "
              "w=" << layerDim.width << ", "
              "h=" << layerDim.height << "), "
              "with rendered width=" << surfaceWidth << ", "
              "rendered height=" << surfaceHeight << ", "
              "blank width=" << blankWidth << ", "
              "blank height=" << blankHeight << ", "
              "pixel format=" << format << ", "
              "bufferDepth=" << bufferDepth);

    start_offset = ((width * surfaceScreenY + surfaceScreenX) * bufferDepth);
    bottomSize = (blankHeight * surfaceDim.width * bufferDepth);
    display_row_width = (m_displayWidth * bufferDepth);
    surface_row_width = (surfaceWidth * bufferDepth);

    status = gcoSURF_Lock(dispSurf, &phyAddr, &logAddr);
    if (status != gcvSTATUS_OK)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to lock surface for gcoSURF=" << dispSurf);
        goto err_surf;
    }

    status = gcoSURF_Unlock(dispSurf, logAddr);
    if (status != gcvSTATUS_OK)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to unlock surface for gcoSURF=" << dispSurf << ", "
                    "logical address=" << logAddr);
        goto err_surf;
    }

    thread = new FileThreadObject(filename,
                                  surfaceWidth,
                                  surfaceHeight,
                                  blankWidth,
                                  blankHeight,
                                  bottomSize,
                                  bufferDepth,
                                  start_offset,
                                  display_row_width,
                                  surface_row_width,
                                  (char*)logAddr,
                                  this,
                                  &m_workerSem);

    startScreenshotThread(thread);

    return;

err_surf:
    delete[] filename;
    sem_post(&m_workerSem);
    return;
}

static gcoSURF getGcoSURFFromSurface(Surface* surface)
{
    if (surface == NULL)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "surface is NULL");
        return 0;
    }
    else
    {
        LOG_DEBUG("Gal2dGraphicSystem",
                  "Called, surface ID=" << surface->getID());
    }

    GcoSurfWaylandPlatformSurface* nativeSurface =
        static_cast<GcoSurfWaylandPlatformSurface*>(surface->platform);
    if (nativeSurface == NULL)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to cast surface->platform to a "
                    "GcoSurfWaylandPlatformSurface, "
                    "surface ID=" << surface->getID() << ", "
                    "surface=" << surface);
        return 0;
    }

    return nativeSurface->m_surface;
}

void Gal2dGraphicSystem::dumpSurfaceToFile(const std::string& fileToSave,
                                           Surface* surface)
{
    gcoSURF surf;
    gcoSURF *tempSurf = new gcoSURF[1];
    gceSTATUS status = gcvSTATUS_OK;
    gctINT stride;
    gctUINT width;
    gctUINT height;
    gceSURF_FORMAT format;
    gctUINT32 phyAddr, surfPhyAddr;
    gctPOINTER logAddr = NULL, surfLogAddr = NULL;

    gcsRECT srcRect;

    uint32_t depthbits;
    size_t buffer_size;
    gceTILING tiling;

    char *filename = new char[fileToSave.length()+1];
    FileThreadObject *thread = NULL;
    strcpy(filename, fileToSave.c_str());

    if (surface == NULL)
    {
        LOG_WARNING("Gal2dGraphicSystem", "surface is NULL");
        goto err_surface;
    }
    else
    {
        LOG_DEBUG("Gal2dGraphicSystem",
                  "Called, file path= " << fileToSave << ", "
                  "surface ID=" << surface->getID() << ", "
                  "surface=" << surface);
    }

    surf = getGcoSURFFromSurface(surface);
    if (!surf)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get gcoSURF from surface, "
                    "ID=" << surface->getID() << ", "
                    "surface=" << surface);
        goto err_surface;
    }
    else
    {
        LOG_DEBUG("Gal2dGraphicSystem",
                  "gcoSURF=" << surf << "for "
                  "surface ID=" << surface->getID() << ", "
                  "surface=" << surface);
    }

    status = gcoSURF_GetAlignedSize(surf, &width, &height, &stride);
    if (status != gcvSTATUS_OK)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get aligned size"
                    "for surface ID=" << surface->getID() << ", "
                    "surface=" << surface << ", "
                    "gcoSURF=" << surf);
        goto err_surface;
    }

    status = gcoSURF_GetFormat(surf, gcvNULL, &format);
    if (status != gcvSTATUS_OK)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get format"
                    "for surface ID=" << surface->getID() << ", "
                    "surface=" << surface << ", "
                    "gcoSURF=" << surf);
        goto err_surface;
    }
    depthbits = getDepthBitsFromFormat(format);
    if (depthbits == 0)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get depth from format, code=" << format << ", "
                    "surface ID=" << surface->getID() << ", "
                    "surface=" << surface << ", "
                    "gcoSURF=" << surf);
        goto err_surface;
    }

    buffer_size = (width * height * depthbits) / 8;

    LOG_DEBUG("Gal2dGraphicSystem",
              "Dumping surface " << surface->getID() << ", "
              "surface=" << surface << ", "
              "pixel format code=" << format << ", "
              "bits per pixel=" << depthbits << ", "
              "width=" << width << ", "
              "height=" << height << ", "
              "buffer size=" << buffer_size);

    status = gcoSURF_Lock(surf, &surfPhyAddr, &surfLogAddr);
    status = gcoSURF_Construct(m_gcoHal,
                               width,
                               height, 1,
                               gcvSURF_BITMAP,
                               format,
                               gcvPOOL_DEFAULT,
                               &tempSurf[0]);

    if (status != gcvSTATUS_OK)
    {
        LOG_DEBUG("Gal2dGraphicSystem",
                  "Failed to construct tempory surface");
        goto err_surface;
    }

    srcRect.left   = (gctINT32)0;
    srcRect.top    = (gctINT32)0;
    srcRect.right  = (gctINT32)width;
    srcRect.bottom = (gctINT32)height;

    status = gcoSURF_Lock(tempSurf[0], &phyAddr, &logAddr);

    gcoSURF_GetTiling(surf, &tiling);

    LOG_INFO("Gal2dGraphicSystem",
                  "Tiling Format, "
                  "surface=" << surface << ", "
                  "surface ID=" << surface->getID() << ", "
                  "Tiling=" << tiling);

    if(gcoHAL_IsFeatureAvailable(m_gcoHal, gcvFEATURE_2D_TILING) != gcvTRUE &&
                    (tiling > gcvLINEAR))
    {
        LOG_WARNING("Gal2dGraphicSystem",
                        "Tiling not supported, but"
                        "surface ID=" << surface->getID() << " is in tiled format");
        goto err_surface;
    }

    status = gco2D_SetGenericSource(m_gco2dEngine, &surfPhyAddr, 1,
                        (gctUINT32_PTR)&stride, 1, tiling, format, gcvSURF_0_DEGREE,
                        width, height);

    status = gco2D_SetSource(m_gco2dEngine, &srcRect);
    if (gcvSTATUS_OK != status)
    {
        goto err_surface;
    }

    status = gco2D_SetTargetEx(m_gco2dEngine, phyAddr, stride,
                        gcvSURF_0_DEGREE, width, height);

    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to set target surface");
        goto err_surface;
    }

    status = gco2D_Blit(m_gco2dEngine, 1, &srcRect, 0xCC, 0xAA, format);
    if (gcvSTATUS_OK != status)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to Blit, "<<
                        "status=" << status << ", "
                        "targetsurface:"<<tempSurf[0]<<","
                        "Destination Area: "
                        "left=" << srcRect.left << ", "
                        "right=" << srcRect.right << ", "
                        "bottom=" << srcRect.bottom << ", "
                        "top=" << srcRect.top<<","
                        "tartget_format="<<format);
        goto err_surface;
    }

    flushAndCommit();

    status = gcoSURF_Unlock(surf, surfLogAddr);
    if (status != gcvSTATUS_OK)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to unlock gcoSURF=" << surf
                    << " for surface ID=" << surface->getID()
                    << ", surface=" << surface);
        goto err_surface;
    }

    status = gcoSURF_Unlock(tempSurf[0], logAddr);
    if (status != gcvSTATUS_OK)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to unlock gcoSURF=" << tempSurf[0]
                    << " for surface ID=" << surface->getID()
                    << ", surface=" << surface);
        goto err_surface;
    }


    thread = new FileThreadObject(filename,
                                  buffer_size,
                                  (char*)logAddr,
                                  this,
                                  &m_workerSem,
                                  tempSurf);

    startScreenshotThread(thread);

    return;

err_surface:
    if(surfLogAddr != NULL)
            gcoSURF_Unlock(surf, surfLogAddr);
    if(logAddr != NULL)
            gcoSURF_Unlock(tempSurf[0], logAddr);
    delete[] filename;
    delete[] tempSurf;
    sem_post(&m_workerSem);
    return;
}

bool Gal2dGraphicSystem::getSurfaceIsRGB(Surface* surface, bool& isRGB)
{
    if (surface == NULL)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "surface is NULL");
        return false;
    }
    else
    {
        LOG_DEBUG("Gal2dGraphicSystem",
                  "Called, surface ID=" << surface->getID() << ", "
                  "surface=" << surface);
    }
    gcoSURF surf = getGcoSURFFromSurface(surface);
    if (!surf)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to get gcoSURF from surface " << surface->getID()
                    << ", surface=" << surface);
        return false;
    }

    gceSTATUS status = gcvSTATUS_OK;
    gceSURF_FORMAT format = gcvSURF_UNKNOWN;
    status = gcoSURF_GetFormat(surf, gcvNULL, &format);
    if (status != gcvSTATUS_OK)
    {
        LOG_WARNING("Gal2dGraphicSystem", "Failed to get format, "
                    "surface ID=" << surface->getID() << ", "
                    "surface=" << surface << ", "
                    "gcoSURF=" << surf);
        return false;
    }

    switch(format)
    {
        case gcvSURF_UNKNOWN:
            LOG_WARNING("Gal2dGraphicSystem", "Pixel format is unknown");
            return false;
        case gcvSURF_A2R2G2B2:
        case gcvSURF_R3G3B2:
        case gcvSURF_A8R3G3B2:
        case gcvSURF_X4R4G4B4:
        case gcvSURF_A4R4G4B4:
        case gcvSURF_R4G4B4A4:
        case gcvSURF_X1R5G5B5:
        case gcvSURF_A1R5G5B5:
        case gcvSURF_R5G5B5A1:
        case gcvSURF_R5G6B5:
        case gcvSURF_R5G5B5X1:
        case gcvSURF_R4G4B4X4:
        case gcvSURF_A4B4G4R4:
        case gcvSURF_A1B5G5R5:
        case gcvSURF_B5G6R5:
        case gcvSURF_B4G4R4A4:
        case gcvSURF_B5G5R5A1:
        case gcvSURF_X4B4G4R4:
        case gcvSURF_X1B5G5R5:
        case gcvSURF_B4G4R4X4:
        case gcvSURF_B5G5R5X1:
        case gcvSURF_R8G8B8:
        case gcvSURF_B8G8R8:
        case gcvSURF_X8R8G8B8:
        case gcvSURF_A8R8G8B8:
        case gcvSURF_R8G8B8A8:
        case gcvSURF_G8R8G8B8:
        case gcvSURF_R8G8B8G8:
        case gcvSURF_X2R10G10B10:
        case gcvSURF_A2R10G10B10:
        case gcvSURF_R8G8B8X8:
        case gcvSURF_X8B8G8R8:
        case gcvSURF_A8B8G8R8:
        case gcvSURF_A2B10G10R10:
        case gcvSURF_B8G8R8X8:
        case gcvSURF_B8G8R8A8:
        case gcvSURF_X2B10G10R10:
        case gcvSURF_X12R12G12B12:
        case gcvSURF_A12R12G12B12:
        case gcvSURF_B16G16R16:
        case gcvSURF_X16R16G16B16:
        case gcvSURF_A16R16G16B16:
        case gcvSURF_X16B16G16R16:
        case gcvSURF_A16B16G16R16:
        case gcvSURF_B32G32R32:
        case gcvSURF_A32R32G32B32:
        case gcvSURF_X32B32G32R32:
        case gcvSURF_A32B32G32R32:
            isRGB = true;
            break;
        default:
            isRGB = false;
            break;
    }
    return true;
}

int  Gal2dGraphicSystem::drmQueryMastership()
{
    int numRetries = 10;
    int waiting_time = 40*1000;/* 40 ms*/

    /* check if there is other process holding semaphore
     * before opening a drm file handle
    */
    m_drm_master_sem = sem_open("/drm_master", O_CREAT, 0666, 1);

    while (m_drm_master_sem == SEM_FAILED || m_drm_master_sem == NULL)
    {
        if ((errno == EACCES) && (numRetries > 0))
        {
            LOG_WARNING("Gal2dGraphicSystem",
                       "could not OPEN a drm_master sem, retry cnt: "<< numRetries);
            usleep(waiting_time);
            numRetries--;
            m_drm_master_sem = sem_open("/drm_master", O_CREAT, 0666, 1);
        }
        else
        {
            LOG_ERROR("Gal2dGraphicSystem",
                       "could not OPEN a drm_master sem giving up, errno: "<< errno
                       << " retry cnt: "<< numRetries);
            return -1;
        }
    }
    /*change the permission, everyone can access
     *ignore the return value:
     *if user root creates a semaphore chmod will fail here*/
    chmod("/dev/shm/sem.drm_master", 0666);

    numRetries = 10;

    while ( sem_trywait(m_drm_master_sem) == -1)
    {
        if (( errno == EAGAIN) && (numRetries > 0))
        {
            /*update timer and restart*/
            LOG_WARNING("Gal2dGraphicSystem",
                        "could not ACQUIERE a drm_master sem, retry cnt: "<< numRetries);
            usleep(waiting_time);
            numRetries--;
        }
        else
        {
            LOG_ERROR("Gal2dGraphicSystem",
                      "could not ACQUIERE a drm_master sem giving up, errno: "<< errno
                       << " retry cnt: "<< numRetries);
            return -1;
        }
    }
    /*success*/
    return 0;
}

bool  Gal2dGraphicSystem::initializeDrmFD()
{
    struct udev* udev;
    struct udev_device *device, *drm_device = NULL;
    struct udev_enumerate* e;
    struct udev_list_entry* entry;
    const char *path, *device_seat;
    const char *seat = default_seat;
    const char* filename;

    udev = udev_new();
    if (udev == NULL)
    {
        LOG_WARNING("Gal2dGraphicSystem",
                    "Failed to initialize udev context.");
        return false;
    }

    e = udev_enumerate_new(udev);
    udev_enumerate_add_match_subsystem(e, "drm");
    udev_enumerate_add_match_sysname(e, "card[0-9]*");
    udev_enumerate_scan_devices(e);

    udev_list_entry_foreach(entry, udev_enumerate_get_list_entry(e))
    {
        path = udev_list_entry_get_name(entry);
        device = udev_device_new_from_syspath(udev, path);
        device_seat = udev_device_get_property_value(device, "ID_SEAT");
        if (!device_seat)
                device_seat = default_seat;
        if (strcmp(device_seat, seat) == 0)
        {
                drm_device = device;
                break;
        }
        udev_device_unref(device);
    }
    udev_enumerate_unref(e);

    if (drm_device != NULL)
    {
        filename = udev_device_get_devnode(drm_device);

        /**
         * Check for the mastership to setup the crtcs in createOutputForConnector below.
         * This may fail due to another process (e.g. v4l2_sink) holding the master ship.
         */
        if (drmQueryMastership() != 0) {
            LOG_WARNING("Gal2dGraphicSystem",
                               "Failed to get the drmMaster");
        }

        m_fdDev = open(filename, O_RDWR | O_CLOEXEC);
        if (m_fdDev >= 0)
        {
            LOG_DEBUG("Gal2dGraphicSystem",
                       "Device name=" << filename << ", fd=" << m_fdDev);

            udev_device_unref(drm_device);
            udev_unref(udev);

            return true;
        }
        else
        {
            LOG_WARNING("Gal2dGraphicSystem",
                        "Failed to open device, name="<< filename);
            udev_device_unref(drm_device);
            udev_unref(udev);

            return false;
        }
    }
    else
    {
        LOG_WARNING("Gal2dGraphicSystem", "No DRM device found");
        udev_unref(udev);

        return false;
    }
}

bool  Gal2dGraphicSystem::setGamma(unsigned int screenID, double value)
{
    bool ret = false;
    DrmOutput* output = NULL;
    int i, r;
    uint16_t lum[GAMMA_ARRAY_SIZE] = {
            0, 2, 4, 8, 16, 32, 64, 96,
            128, 160, 192, 224, 256, 288, 320, 352, 383
            };
    float center, width, start, height, t, scurve[GAMMA_ARRAY_SIZE], gammacurve[GAMMA_ARRAY_SIZE];

    wl_list_for_each(output, &m_outputList, link)
    {
        if (output->screenID == screenID)
        {
            break;
        }
    }

    center = 32; /* nominal 32 - must be in {1,380} where is s-curve centered*/
    width = 32; /* nominal 32 - must be in {1,380} how narrow is s-curve */
    for (i = 0; i < GAMMA_ARRAY_SIZE; i++) {
        t = (float)lum[i];
        scurve[i] = (256.F/3.14159F)*((float) atan((t-center)/width));
        gammacurve[i] = 255.F* ((float) pow(t/384.F,1.0/value));
    }

    start = scurve[0];
    height = scurve[GAMMA_ARRAY_SIZE-1] - start;
    for(i = 0; i < GAMMA_ARRAY_SIZE; i++) {
        scurve[i] = (256.F/height)*(scurve[i]-start);
        lum[i] = (int)((scurve[i] * (gammacurve[i]/256.F)) + 0.5);
    }

    if(output)
    {
        r = drmModeCrtcSetGamma(m_fdDev, output->crtcID,DRM_IMX_GAMMA_SIZE,lum, lum, lum);
        LOG_INFO("Gal2dGraphicSystem",
                "drmModeCrtcSetGamma executed on crtc: "<< output->crtcID<<" and returned with: "<<r);
        LOG_INFO("Gal2dGraphicSystem","lum 0: "<<lum[0] <<" 1:"<<lum[1]
                                                        <<" 2:"<<lum[2]
                                                        <<" 3:"<<lum[3]
                                                        <<" 4:"<<lum[4]
                                                        <<" 5:"<<lum[5]
                                                        <<" 6:"<<lum[6]
                                                        <<" 7:"<<lum[7]
                                                        <<" 8:"<<lum[8]
                                                        <<" 9:"<<lum[9]
                                                        <<" 10:"<<lum[10]
                                                        <<" 11:"<<lum[11]
                                                        <<" 12:"<<lum[12]
                                                        <<" 13:"<<lum[13]
                                                        <<" 14:"<<lum[14]
                                                        <<" 15:"<<lum[15]);
        if (r)
            ret = true;
    }

    return ret;
}

bool Gal2dGraphicSystem::setCSC(unsigned int screenID, ilmCSCProperties* pclrprop)
{
    DrmOutput* output = NULL;
    int ret;
    int i;
    int j;
    ilmUSERSpecificCSCvalues user_cscvalues= {ILM_INVALID_CSC_MATRIX_VALUE,
            ILM_INVALID_CSC_MATRIX_VALUE,
            ILM_INVALID_CSC_MATRIX_VALUE,
            ILM_INVALID_CSC_MATRIX_VALUE,
            ILM_INVALID_CSC_MATRIX_VALUE,
            ILM_INVALID_CSC_MATRIX_VALUE};

    if(NULL == pclrprop)
        return false;

    wl_list_for_each(output, &m_outputList, link)
    {
        if (output->screenID == screenID)
        {
            break;
        }
    }

    if(NULL == output)
	return false;

    mConfiguration->getCSCValues(screenID,
                                    &user_cscvalues);

    if(true == pclrprop->set_to_default)
    {
        csc_set_default_value(&output->clrprop);
        output->clrprop.csc.type = DRM_IMX_CSC_RGB2RGB;
    }
    else
    {
        if(pclrprop->hue != ILM_INVALID_CSC_VALUE)
                output->clrprop.hue = pclrprop->hue;

        if(pclrprop->saturation != ILM_INVALID_CSC_VALUE)
                output->clrprop.saturation = pclrprop->saturation;

        if(pclrprop->brightness != ILM_INVALID_CSC_VALUE)
                output->clrprop.brightness = pclrprop->brightness;

        if(pclrprop->contrast != ILM_INVALID_CSC_VALUE)
                output->clrprop.contrast = pclrprop->contrast;

        if(pclrprop->hue_off != ILM_INVALID_CSC_VALUE)
                output->clrprop.hue_off = pclrprop->hue_off;

        if(pclrprop->saturation_off != ILM_INVALID_CSC_VALUE)
                output->clrprop.saturation_off = pclrprop->saturation_off;

        if(pclrprop->brightness_off != ILM_INVALID_CSC_VALUE)
                output->clrprop.brightness_off = pclrprop->brightness_off;

        switch (pclrprop->type)
        {
                case ILM_CSC_TYPE_RGB2RGB:
                        output->clrprop.csc.type = DRM_IMX_CSC_RGB2RGB;
                        break;

                case ILM_CSC_TYPE_RGB2YUV:
                        output->clrprop.csc.type = DRM_IMX_CSC_RGB2YUV;
                        break;

                case ILM_CSC_TYPE_YUV2RGB:
                        output->clrprop.csc.type = DRM_IMX_CSC_YUV2RGB;
                        break;

                case ILM_CSC_TYPE_YUV2YUV:
                        output->clrprop.csc.type = DRM_IMX_CSC_YUV2YUV;
                        break;

                case ILM_CSC_TYPE_INVALID:
                default:
                        output->clrprop.csc.type = DRM_IMX_CSC_RGB2RGB;
                        break;
        }
    }

    LOG_INFO("Gal2dGraphicSystem","screenID: "<<screenID << "hue: "<< output->clrprop.hue <<
                                    " saturation:"<<output->clrprop.saturation <<
                                    " brightness:"<<output->clrprop.brightness <<
                                    " contrast:"<<output->clrprop.contrast <<
                                    " hue_off:"<<output->clrprop.hue_off <<
                                    " saturation_off:"<<output->clrprop.saturation_off <<
                                    " brightness_off:"<<output->clrprop.brightness_off <<
                                    " type:"<<output->clrprop.csc.type);

    ret = csc_dp_compute_update_matrix(m_fdDev, false, &output->clrprop);
    if(ret < 0)
    {
        LOG_ERROR("Gal2dGraphicSystem", "csc_dp_compute_update_matrix failed");
        return false;
    }

    LOG_INFO("Gal2dGraphicSystem","A Matrix Values:");
    for(i = 0; i < 3; i++)
    {
        for(j = 0; j < 3; j++)
        {
            LOG_INFO("Gal2dGraphicSystem","A[" << i << "][" << j << "]:" << output->clrprop.csc.A[i][j]);
        }
    }

    LOG_INFO("Gal2dGraphicSystem","B Offset Vector");
    for(i = 0; i < 3; i++)
    {
        LOG_INFO("Gal2dGraphicSystem","B[" << i << "]:" << output->clrprop.csc.B[i]);
    }

    LOG_INFO("Gal2dGraphicSystem","E Exponent Vector\n");
    for(i = 0; i < 3; i++)
    {
        LOG_INFO("Gal2dGraphicSystem","E[" << i << "]:" << output->clrprop.csc.E[i]);
    }

    LOG_INFO("Gal2dGraphicSystem", "scaling to user specific csc matrix values (if provided)");
    for (i = 0; i < 3; i++)
    {
        if(ILM_INVALID_CSC_MATRIX_VALUE != user_cscvalues.diagnal_matrix[i])
        {
            output->clrprop.csc.A[i][i] *= (double)user_cscvalues.diagnal_matrix[i]/ILM_CSC_MATRIX_MAX_VALUE;
            if(ILM_CSC_MATRIX_MAX_VALUE < output->clrprop.csc.A[i][i]) {
                output->clrprop.csc.A[i][i] = ILM_CSC_MATRIX_MAX_VALUE;
            }

            LOG_INFO("Gal2dGraphicSystem","A Matrix Values [" << i << "][" << i << "]:" << output->clrprop.csc.A[i][i]);
        }
    }

    for (i = 0; i < 3; i++)
    {
        if(ILM_INVALID_CSC_MATRIX_VALUE != user_cscvalues.offset_vector[i])
        {
            output->clrprop.csc.B[i] += user_cscvalues.offset_vector[i];
            if(ILM_CSC_B_OFFSET_SIGNED_FOURTEEN_BIT_VAL < output->clrprop.csc.B[i])
                output->clrprop.csc.B[i] = ILM_CSC_B_OFFSET_SIGNED_FOURTEEN_BIT_VAL;

            LOG_INFO("Gal2dGraphicSystem","B Offset vector [" << i << "]:" << output->clrprop.csc.B[i]);
        }
    }

    ret = drmIoctl(m_fdDev, DRM_IOCTL_IMX_SET_CSC, &output->clrprop.csc);
    if(ret < 0)
    {
        LOG_ERROR("Gal2dGraphicSystem", "DRM_IOCTL_IMX_SET_CSC failed");
        return false;
    }

    return true;
}

bool Gal2dGraphicSystem::getDPMSValue(ilmScreenState screenState, int& value)
{
	bool success = true;
	int dpmsValue = -1;
	switch (screenState)
	{
		case ILM_SCREEN_ON:
			dpmsValue = DRM_MODE_DPMS_ON;
			break;
		case ILM_SCREEN_STANDBY:
			dpmsValue = DRM_MODE_DPMS_STANDBY;
			break;
		case ILM_SCREEN_SUSPEND:
			dpmsValue = DRM_MODE_DPMS_SUSPEND;
			break;
		case ILM_SCREEN_OFF:
			dpmsValue = DRM_MODE_DPMS_OFF;
			break;
		default:
			success = false;
			LOG_ERROR("Gal2dGraphicSystem", "Unknown screen state = " << screenState);
			break;
	}
	value = dpmsValue;
	return success;
}

bool Gal2dGraphicSystem::setDisplayState(unsigned int screenId, ilmScreenState screenState)
{
    DrmOutput *output = NULL;
    bool success = false;
    int dpmsValue = -1;
    int ret = -1;
    uint64_t fd_value = 1ull;
    LOG_DEBUG("Gal2dGraphicSystem", "setDisplayState called screenID = "
                                << screenId << "screen state = " << screenState);
    wl_list_for_each(output, &m_outputList, link)
    {
        if (output->screenID == screenId)
        {
            if (NULL != output->propDPMS)
            {
                if (getDPMSValue(screenState, dpmsValue))
                {
                    pthread_mutex_lock(&m_graSystemMutex);
                    if(false == output->dpms_readytoChange)
                    {
                        if((output->dpms_current_state != dpmsValue))
                        {
                            output->dpms_newState= dpmsValue;
                            output->dpms_readytoChange = true;
                            ret = write(dpms_fdDev, &fd_value, sizeof(uint64_t));
                            if(ret == -1)
                            {
                                LOG_ERROR("Gal2dGraphicSystem", "FAILED to  Write to the dpms_fdDev");
                            }
                            else
                            {
                                success = true;
                            }
                        }
                        else
                        {
                            LOG_DEBUG("Gal2dGraphicSystem", "Display state is same as previous state");
                        }
                    }
                    else
                    {
                        if(output->dpms_current_state == dpmsValue)
                        {
                            output->dpms_readytoChange = false;
                        }
                        output->dpms_newState = dpmsValue;
                    }
                    pthread_mutex_unlock(&m_graSystemMutex);
                }
            }
            else
            {
                LOG_ERROR("Gal2dGraphicSystem",
                        "dpms property unavailable for screen = " << screenId
                        << " connector_id = " << output->connectorID);
            }
        }
    }
    return success;
}

bool Gal2dGraphicSystem::getILMScreenValue(ilmScreenState* screenState, int value)
{
    bool success = true;
    ilmScreenState ilmscreenValue = ILM_SCREEN_UNKNOWN;
    switch (value)
    {
        case DRM_MODE_DPMS_ON:
            ilmscreenValue = ILM_SCREEN_ON;
            break;
        case DRM_MODE_DPMS_STANDBY:
            ilmscreenValue = ILM_SCREEN_STANDBY;
            break;
        case DRM_MODE_DPMS_SUSPEND:
            ilmscreenValue = ILM_SCREEN_SUSPEND;
            break;
        case DRM_MODE_DPMS_OFF:
            ilmscreenValue = ILM_SCREEN_OFF;
            break;
        default:
            success = false;
            break;
    }
    *screenState = ilmscreenValue;
    return success;
}


bool Gal2dGraphicSystem::getDisplayState(unsigned int screenId, ilmScreenState *screenState)
{
    DrmOutput *output = NULL;
    bool ret = false;
    LOG_DEBUG("Gal2dGraphicSystem", "getDisplayState called screenID = "
                                        << screenId );
    wl_list_for_each(output, &m_outputList, link)
    {
        if (output->screenID == screenId)
        {
            if (NULL != output->propDPMS)
            {
                if(ret == getILMScreenValue( screenState, output->dpms_current_state))
                {
                    LOG_ERROR("Gal2dGraphicSystem", "Unknown dpms state = " << output->dpms_current_state);
                }
                else
                {
                    ret = true;
                    LOG_DEBUG("Gal2dGraphicSystem", "screenID = "<< screenId << "state is " << screenState[0]
                                                     << " for dpms state " << output->dpms_current_state << "\n");
                }
            }
            else
            {
                LOG_ERROR("Gal2dGraphicSystem",
                        "dpms property unavailable for screen = " << screenId
                        << " connector_id = " << output->connectorID);
            }
        }
    }
    return ret;
}

bool Gal2dGraphicSystem::setDisplayAlpha(unsigned int screenId, unsigned int alphaValue)
{
    DrmOutput *output = NULL;
    bool ret = false;
    LOG_DEBUG("Gal2dGraphicSystem", "setDisplayAlpha called screenID = "
                                << screenId << "alpha value = " << alphaValue);
    wl_list_for_each(output, &m_outputList, link)
    {
        if (output->screenID == screenId)
        {
            if (alphaValue <= ALPHA_MAX)
            {
                LOG_DEBUG("Gal2dGraphicSystem",
                        "setting alpha value "<< alphaValue
						<< " for screenID = " << screenId);
                output->screen_alpha = (alphaValue / 255.0f);
                ret = true;
            }
            else
            {
                LOG_ERROR("Gal2dGraphicSystem","Try to set alpha value "
                        <<alphaValue
						<<" for screenID = "
                        << screenId
						<< " Failed ,set alpha value between 0 to 255");
            }
        }
    }
    return ret;
}

bool Gal2dGraphicSystem::getDisplayAlpha(unsigned int screenId, unsigned int *alphaValue)
{
    DrmOutput *output = NULL;
    bool ret = false;
    LOG_DEBUG("Gal2dGraphicSystem", "getDisplayAlpha called screenID = "
                                        << screenId );
    wl_list_for_each(output, &m_outputList, link)
    {
        if (output->screenID == screenId)
        {
                alphaValue[0] = (output->screen_alpha * 255);
                LOG_DEBUG("Gal2dGraphicSystem", "Alpha value :" << alphaValue[0]);
                ret = true;
        }
    }
    return ret;
}
